<?php

namespace App\Controller\Admin\Device;

use App\Bundles\Sg\DatatablesBundle\Response\DatatableQueryBuilder;
use App\Bundles\Sg\DatatablesBundle\Response\DatatableResponse;
use App\Controller\Admin\BaseController;
use App\Entity\AgentProfile\AgentProfile;
use App\Entity\Alert\Alert;
use App\Entity\ConfigProfile\ConfigProfile;
use App\Entity\Device\Device;
use App\Entity\Snmp\Device as SnmpDevice;
use App\Entity\Device\DeviceOsVersion;
use App\Entity\Device\DeviceType;
use App\Entity\Label\Label;
use App\Entity\Label\LabelDeviceMap;
use App\Entity\Label\SystemLabel;
use App\Entity\Label\SystemLabelDeviceMap;
use App\Entity\MonitoringGroup\MonitoringGroup;
use App\Entity\MonitoringGroup\MonitoringGroupMapping;
use App\Entity\Tool\AuthProfile;
use App\Exception\AlertNotFoundException;
use App\Exception\DeviceNotFoundException;
use App\Exception\DeviceNumberInvalidException;
use App\Exception\MonitoringGroupNotFoundException;
use App\Form\DeleteType;
use App\Form\Device\DeviceAgentProfileType;
use App\Form\Device\DeviceAuthProfileType;
use App\Form\Device\DeviceConfigProfileType;
use App\Form\Device\DeviceCustomLabelType;
use App\Form\Device\DeviceCustomModuleDownloadType;
use App\Form\Device\DeviceHostnameType;
use App\Form\Device\DeviceNetDevicesType;
use App\Form\Device\DeviceSystemLabelType;
use App\Form\Device\FwVersionType;
use App\Form\Device\MonitoringGroupType;
use App\Repository\ConfigProfileRepository;
use App\Repository\DeviceRepository;
use App\Repository\DeviceUserModuleRepository;
use App\Repository\MonitoringGroupRepository;
use App\Repository\ProductFlagsRepository;
use App\Service\Api\AgentProfiles;
use App\Service\Api\ApiService;
use App\Service\Api\ConfigProfiles;
use App\Service\Api\DashboardLights;
use App\Service\Command;
use App\Service\DatatableService;
use App\Service\ExportService;
use App\Service\History;
use Doctrine\DBAL\Exception;
use Doctrine\ORM\EntityManagerInterface;
use Doctrine\Persistence\ManagerRegistry;
use Psr\Log\LoggerInterface;
use Symfony\Component\DependencyInjection\ParameterBag\ParameterBagInterface;
use Symfony\Component\Form\FormInterface;
use Symfony\Component\HttpFoundation\BinaryFileResponse;
use Symfony\Component\HttpFoundation\JsonResponse;
use Symfony\Component\HttpFoundation\RedirectResponse;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\HttpFoundation\Response;
use Symfony\Component\HttpFoundation\ResponseHeaderBag;
use Symfony\Component\Routing\Annotation\Route;
use Symfony\Component\Serializer\Encoder\CsvEncoder;
use Symfony\Component\Serializer\Normalizer\ObjectNormalizer;
use Symfony\Component\Serializer\Serializer;

#[Route(path: '/admin/devices', name: 'admin_devices')]
class DeviceController extends BaseController
{

    private ?Response $render = null;

    /**
     * @param ManagerRegistry $managerRegistry
     * @param EntityManagerInterface $entityManager
     * @param DeviceRepository $deviceRepository
     * @param ProductFlagsRepository $productFlagsRepository
     */
    public function __construct(ManagerRegistry                         $managerRegistry,
								private readonly EntityManagerInterface $entityManager,
								private readonly DeviceRepository       $deviceRepository,
								private readonly ProductFlagsRepository $productFlagsRepository, private readonly LoggerInterface $logger) {

        parent::__construct($managerRegistry);

    }

    /**
     * @param Request $request
     * @param DatatableResponse $responseService
     * @param DatatableService $datatableService
     * @return JsonResponse|Response
     * @throws \Exception
     */
    #[Route(path: '', name: '')]
    public function listDevices(Request $request, DatatableResponse $responseService, DatatableService $datatableService): JsonResponse|Response
    {

        $isAjax = $request->isXmlHttpRequest();
        $this->filter = $request->get('column');

        $datatable = $datatableService->getDeviceDatatable();

        if ($isAjax) {
            $responseService->setDatatable($datatable);
            //$responseService->getDatatableQueryBuilder();

            //$datatableQueryBuilder = $responseService->getDatatableQueryBuilder();

            $requestParams = $this->getDatatableFilter($request, $datatable);
            $datatableQueryBuilder = new DatatableQueryBuilder($requestParams, $datatable);
            $responseService->setDatatableQueryBuilder($datatableQueryBuilder);

            $qb = $datatableQueryBuilder->getQb();
            $qb->andWhere('vdevice.deleted = :deleted');
            $qb->setParameter('deleted', '0');

            return $responseService->getResponse();
        }

        return $this->render('device/index.html.twig', [
            'datatable' => $datatable,
            'filter' => $this->filter,
            'title' => 'Registered Devices'
        ]);

    }

    /**
     * @param int $alertId
     * @param Request $request
     * @param ManagerRegistry $managerRegistry
     * @param ApiService $apiService
     * @param DatatableResponse $responseService
     * @param DatatableService $datatableService
     * @return JsonResponse|Response
     * @throws AlertNotFoundException
     */
    #[Route(path: '/alert/{alertId}', name: '_triggered_alert')]
    public function listDevicesByAlertId(int $alertId, Request $request, ManagerRegistry $managerRegistry, ApiService $apiService,
                                         DatatableResponse $responseService, DatatableService $datatableService): JsonResponse|Response
    {

        $isAjax = $request->isXmlHttpRequest();
        $this->filter = $request->get('column');

        /** @var Alert $alert */
        $alert = $managerRegistry->getRepository(Alert::class)->findOneBy(['id' => $alertId]);

        if(!$alert){

            throw new AlertNotFoundException();

        }

        $trigerredAlertDevices = $apiService->getTriggeredAlertDevices($alertId);

        $datatable = $datatableService->getDeviceDatatable();

        if ($isAjax) {
            $responseService->setDatatable($datatable);
            $requestParams = $this->getDatatableFilter($request, $datatable);
            $datatableQueryBuilder = new DatatableQueryBuilder($requestParams, $datatable);
            $responseService->setDatatableQueryBuilder($datatableQueryBuilder);

            /** @var QueryBuilder $qb */
            $qb = $datatableQueryBuilder->getQb();
            $qb->andWhere('vdevice.id IN (:deviceIds)');
            $qb->andWhere('vdevice.deleted = :deleted');
            $qb->setParameter('deviceIds', $trigerredAlertDevices);
            $qb->setParameter('deleted', '0');

            return $responseService->getResponse();
        }

        $alertUrl = $this->generateUrl('admin_alert_edit', [
            'alertId' => $alert->getId(),
        ]);

        return $this->render('device/index.html.twig', ['datatable' => $datatable, 'filter' => $this->filter, 'title' => 'List of devices for trigerred alert: <a href="' . $alertUrl . '">' . $alert->getName() . '</a>']);

    }

	/**
	 * @param int $monitoringGroupId
	 * @param int $subsetId
	 * @param Request $request
	 * @param ManagerRegistry $managerRegistry
	 * @param DashboardLights $dashboardLightsApi
	 * @param DatatableResponse $responseService
	 * @param DatatableService $datatableService
	 * @return JsonResponse|Response
	 * @throws MonitoringGroupNotFoundException
	 */
    #[Route(path: '/dashboard/monitoring-group/{monitoringGroupId}/subset/{subsetId}', name: '_dashboard_light')]
    public function listDevicesByDashboardLight(int $monitoringGroupId, int $subsetId, Request $request,
                                                ManagerRegistry $managerRegistry, DashboardLights $dashboardLightsApi,
                                         DatatableResponse $responseService, DatatableService $datatableService): JsonResponse|Response
    {

        $isAjax = $request->isXmlHttpRequest();
        $this->filter = $request->get('column');

        /** @var MonitoringGroup $monitoringGroup */
        $monitoringGroup = $managerRegistry->getRepository(MonitoringGroup::class)->findOneBy(['id' => $monitoringGroupId]);

        if(!$monitoringGroup){

            throw new MonitoringGroupNotFoundException();

        }

        $triggeredDevices = $dashboardLightsApi->getDashboardDetailData($monitoringGroup->getId(), $subsetId);

        $datatable = $datatableService->getDeviceDatatable();

        if ($isAjax) {
            $responseService->setDatatable($datatable);
            $requestParams = $this->getDatatableFilter($request, $datatable);
            $datatableQueryBuilder = new DatatableQueryBuilder($requestParams, $datatable);
            $responseService->setDatatableQueryBuilder($datatableQueryBuilder);

            $qb = $datatableQueryBuilder->getQb();
            $qb->andWhere('vdevice.id IN (:deviceIds)');
            $qb->andWhere('vdevice.deleted = :deleted');
            $qb->setParameter('deviceIds', $triggeredDevices);
            $qb->setParameter('deleted', '0');

            return $responseService->getResponse();
        }

        if($subsetId === 0){
            $dashboardDeviceStatusType = 'Online';
        }elseif ($subsetId === 1){
            $dashboardDeviceStatusType = 'Delayed';
        }elseif ($subsetId === 2){
            $dashboardDeviceStatusType = 'Late';
        }elseif ($subsetId === 3){
            $dashboardDeviceStatusType = 'Synced';
        }elseif ($subsetId === 4){
            $dashboardDeviceStatusType = 'Pending';
        }elseif ($subsetId === 5){
            $dashboardDeviceStatusType = 'Error';
        }else{
            $dashboardDeviceStatusType = '';
        }

        return $this->render('device/index.html.twig', [
            'datatable' => $datatable,
            'filter' => $this->filter,
            //'title' => 'List of devices for trigerred alert: <a href="' . $alertUrl . '">' . $alert->getName() . '</a>'
            'title' => 'List of ' . $dashboardDeviceStatusType .' devices:',
        ]);

    }

    /**
     * @param int $alertId
     * @param Request $request
     * @param ManagerRegistry $managerRegistry
     * @param ApiService $apiService
     * @param DatatableResponse $responseService
     * @param DatatableService $datatableService
     * @return JsonResponse|Response|null
     * @throws AlertNotFoundException
     */
    #[Route(path: '/past-alert/{alertId}', name: '_past_alert')]
    public function listDevicesByPastAlertId(int $alertId, Request $request, ManagerRegistry $managerRegistry, ApiService $apiService,
                                         DatatableResponse $responseService, DatatableService $datatableService): JsonResponse|Response|null
    {

        $isAjax = $request->isXmlHttpRequest();
        $this->filter = $request->get('column');

        /** @var Alert $alert */
        $alert = $managerRegistry->getRepository(Alert::class)->findOneBy(['id' => $alertId]);

        if(!$alert){

            throw new AlertNotFoundException();

        }

        $pastAlertDevices = $apiService->getPastAlertDevices($alertId);

        $datatable = $datatableService->getDeviceDatatable();

        if ($isAjax) {
            $responseService->setDatatable($datatable);
            $requestParams = $this->getDatatableFilter($request, $datatable);
            $datatableQueryBuilder = new DatatableQueryBuilder($requestParams, $datatable);
            $responseService->setDatatableQueryBuilder($datatableQueryBuilder);

            $qb = $datatableQueryBuilder->getQb();
            $qb->andWhere('vdevice.id IN (:deviceIds)');
            $qb->andWhere('vdevice.deleted = :deleted');
            $qb->setParameter('deviceIds', $pastAlertDevices);
            $qb->setParameter('deleted', '0');

            return $responseService->getResponse();
        }

        $alertUrl = $this->generateUrl('admin_alert_edit', [
            'alertId' => $alert->getId(),
        ]);

        $this->render = $this->render('device/index.html.twig', [
			'datatable' => $datatable,
			'filter' => $this->filter,
			'title' => 'List of devices for past alert: <a href="' . $alertUrl . '">' . $alert->getName() . '</a>'
		]);
        return $this->render;

    }

    /**
     * @throws \Exception
     */
    #[Route(path: '/data-download', name: '_data_download')]
    public function dataDownload(ExportService $exportService): Response
    {

        $exportData = $exportService->getDataExportForDeviceDataTable();

        $csvData = [];
        $csvData[] = array_keys($exportData[0]);

        foreach($exportData as $line){
            $csvData[] = array_values($line);
        }

        $encoders = [new CsvEncoder(['no_headers' => true ])];
        $normalizers = array(new ObjectNormalizer());
        $serializer = new Serializer($normalizers, $encoders);
        $csvContent = $serializer->serialize($csvData, 'csv');

        $fileName = 'DeviceDwarfguardExport_' . date('d-m-Y_H-i');

        $response = new Response($csvContent);
        $response->headers->set('Content-Encoding', 'UTF-8');
        $response->headers->set('Content-Type', 'text/csv; charset=UTF-8');
        $response->headers->set('Content-Disposition', 'attachment; filename='. $fileName .'.csv');
        return $response;

    }

	/**
	 * @param int $deviceId
	 * @param ManagerRegistry $managerRegistry
	 * @param DeviceRepository $deviceRepository
	 * @param ConfigProfileRepository $configProfileRepository
	 * @param History $historyService
	 * @param Command $commandService
	 * @param ParameterBagInterface $params
	 * @param ApiService $apiService
	 * @param Request $request
	 * @param DeviceUserModuleRepository $userModuleRepository
	 * @param MonitoringGroupRepository $monitoringGroupRepository
	 * @param AgentProfiles $agentProfilesApi
	 * @param ConfigProfiles $configProfilesApi
	 * @return BinaryFileResponse|RedirectResponse|Response
	 * @throws DeviceNotFoundException
	 * @throws DeviceNumberInvalidException
	 * @throws Exception
	 * @throws \JsonException
	 */
    #[Route(path: '/{deviceId}', name: '_detail')]
    public function deviceDetail(
        int $deviceId,
        ManagerRegistry $managerRegistry,
        DeviceRepository $deviceRepository,
        ConfigProfileRepository $configProfileRepository,
        History $historyService,
        Command $commandService,
        ParameterBagInterface $params,
        ApiService $apiService,
        Request $request,
        DeviceUserModuleRepository $userModuleRepository,
        MonitoringGroupRepository $monitoringGroupRepository,
        AgentProfiles $agentProfilesApi,
		ConfigProfiles $configProfilesApi
		)
    {

        if (!is_numeric($deviceId)) {

            throw new DeviceNumberInvalidException();

        }

        /** @var Device $device */
        $device = $managerRegistry->getRepository(Device::class)->findOneBy(['id' => $deviceId]);

        if (!$device || $device->isDeleted()) {

            throw new DeviceNotFoundException();

        }

        $findBy = ['available' => 1, 'approved' => 1];

        if (!is_null($device->getOsVer())) {

            $findBy['osType'] = $device->getOsVer()->getOsType();

        }

        $osVersions = $managerRegistry->getRepository(DeviceOsVersion::class)->findBy($findBy, ['version' => 'DESC']);
		$configProfiles = $this->managerRegistry->getRepository(ConfigProfile::class)->findBy(['directive' => 1, 'deleted' => 0]);

        $fwVersionForm = $this->createForm(FwVersionType::class, ['osVersions' => $osVersions, 'showNoSpecific' => !is_null($device->getOsVerDesired()), 'device' => $device]);
        $deleteForm = $this->createForm(DeleteType::class, ['text' => 'Delete device']);
		$configProfileForm = $this->createForm(DeviceConfigProfileType::class, [
			'device' => $device, 'configProfiles' => $configProfiles, 'isAssigned' => ( ($device->getConfigProfile()) ? true : false )]);
		$hostnameForm = $this->createForm(DeviceHostnameType::class, ['device' => $device]);
        $customModuleDownloadForm = $this->createForm(DeviceCustomModuleDownloadType::class, ['device' => $device]);
        $netDevicesForm = $this->createForm(DeviceNetDevicesType::class, ['netDevices' => $device->getNetDevices()]);

        [$labelsForm, $systemLabelsForm] = $this->getLabelForms($managerRegistry, $device, $request);

        $monitoringGroups = $monitoringGroupRepository->getAllMonitoringGroups();
        $deviceMonitoringGroups = $device->monitoringGroupMappings;
        $monitoringGroupsForm = $this->createForm(MonitoringGroupType::class, ['deviceMonitoringGroups' => $deviceMonitoringGroups, 'monitoringGroups' => $monitoringGroups]);

        $agentProfiles = $managerRegistry->getRepository(AgentProfile::class)->findAll();
        $agentProfilesForm = $this->createForm(DeviceAgentProfileType::class, ['device' => $device, 'agentProfiles' => $agentProfiles]);

        $authProfiles = $managerRegistry->getRepository(AuthProfile::class)->findAll();
        $authProfilesForm = $this->createForm(DeviceAuthProfileType::class, ['device' => $device, 'authProfiles' => $authProfiles]);

        $packages = $userModuleRepository->getDeviceUserModules($device);

        $fwVersionForm->handleRequest($request);
        $deleteForm->handleRequest($request);
        $hostnameForm->handleRequest($request);
        $monitoringGroupsForm->handleRequest($request);
		$configProfileForm->handleRequest($request);
        $agentProfilesForm->handleRequest($request);
        $authProfilesForm->handleRequest($request);
        $customModuleDownloadForm->handleRequest($request);
        $netDevicesForm->handleRequest($request);

        $historyTypeId = 1;
        $historyTimeLegend = '"' . implode('","', $historyService->getHistoryTimeLegend($historyTypeId)) . '"';
        $historyData = $historyService->getHistoryDataForDevice($historyTypeId, $device->getId());

        if ($deleteForm->isSubmitted() && $deleteForm->isValid()) {
			return $this->processDeleteFormSubmit($managerRegistry, $apiService, $device);
		}

        if ($customModuleDownloadForm->isSubmitted() && $customModuleDownloadForm->isValid()) {
			return $this->processCustomModuleDownloadFormSubmit($customModuleDownloadForm, $commandService, $managerRegistry, $params, $device);
		}

        if ($hostnameForm->isSubmitted() && $hostnameForm->isValid()) {
			return $this->processHostnameFormSubmit($hostnameForm, $device, $managerRegistry);
		}

        if ($labelsForm->isSubmitted() && $labelsForm->isValid()) {
            return $this->processCustomLabelsFormSubmit($labelsForm, $managerRegistry, $device);
        }

        if ($systemLabelsForm->isSubmitted() && $systemLabelsForm->isValid()) {
            return $this->processSystemLabelsFormSubmit($systemLabelsForm, $managerRegistry, $device);
        }

        if ($netDevicesForm->isSubmitted() && $netDevicesForm->isValid()) {
            return $this->processNetDevicesFormSubmit($netDevicesForm, $device, $managerRegistry);
        }

        if ($monitoringGroupsForm->isSubmitted() && $monitoringGroupsForm->isValid()) {
            return $this->processMonitoringGroupsFormSubmit($monitoringGroupsForm, $managerRegistry, $device);
        }

        if ($fwVersionForm->isSubmitted() && $fwVersionForm->isValid()){
            return $this->processFwVersionFormSubmit($fwVersionForm, $device, $managerRegistry);
        }

		if ($configProfileForm->isSubmitted() && $configProfileForm->isValid()){
			return $this->processConfigProfileFormSubmit($configProfileForm, $device, $managerRegistry, $configProfilesApi);
		}

        if ($agentProfilesForm->isSubmitted() && $agentProfilesForm->isValid()){
			return $this->processAgentProfileFormSubmit($agentProfilesForm, $managerRegistry, $agentProfilesApi, $device);
		}

        if ($authProfilesForm->isSubmitted() && $authProfilesForm->isValid()){
			return $this->processAuthProfileFormSubmit($authProfilesForm, $managerRegistry, $device, $deviceId);
		}

        return $this->render('device/detail.html.twig', [
            'device' => $device,
            'hostnameForm' => $hostnameForm->createView(),
            'fwForm' => $fwVersionForm->createView(),
            'deleteForm' => $deleteForm->createView(),
			'configProfilesForm' => $configProfileForm->createView(),
            'customModuleDownloadForm' => $customModuleDownloadForm->createView(),
            'labelsForm' => $labelsForm->createView(),
            'systemLabelsForm' => $systemLabelsForm->createView(),
            'monitoringGroupsForm' => $monitoringGroupsForm->createView(),
            'agentProfilesForm' => $agentProfilesForm->createView(),
            'authProfilesForm' => $authProfilesForm->createView(),
            'netDevicesForm' => $netDevicesForm->createView(),
            'packages' => $packages,
            'historyTimeLegend' => $historyTimeLegend,
            'osVersions' => $osVersions,
            'historyData' => json_encode($historyData, JSON_THROW_ON_ERROR/*, JSON_PRETTY_PRINT | ENT_QUOTES*/)
        ]);

    }

    /**
     * @param int|null $configProfileId
     * @param array $devicesId
     * @return bool
     * @throws Exception
     */
    private function updateDevicesConfigProfile(?int $configProfileId, array $devicesId): bool {

        try{

            $this->entityManager->getConnection()->beginTransaction();
            $this->deviceRepository->updateDevicesConfigProfile($configProfileId, $devicesId);
            $this->productFlagsRepository->incrementDevchgVersion();
            $this->entityManager->getConnection()->commit();

            return true;

        }catch (\Exception){

            $this->entityManager->getConnection()->rollBack();
            return false;

        }

    }

    /**
     * @param ManagerRegistry $managerRegistry
     * @param Device $device
     * @param Request $request
     * @return array
     */
    private function getLabelForms(ManagerRegistry $managerRegistry, Device $device, Request $request): array
    {
        $customLabels = $managerRegistry->getRepository(Label::class)->findAll();
        $systemLabels = $managerRegistry->getRepository(SystemLabel::class)->findAll();
        $deviceLabels = $device->labels;
        $deviceSystemLabels = $device->systemLabels;
        $labelsForm = $this->createForm(DeviceCustomLabelType::class, ['deviceLabels' => $deviceLabels, 'labels' => $customLabels]);
        $systemLabelsForm = $this->createForm(DeviceSystemLabelType::class, ['deviceLabels' => $deviceSystemLabels, 'labels' => $systemLabels]);
        $labelsForm->handleRequest($request);
        $systemLabelsForm->handleRequest($request);
        return array($labelsForm, $systemLabelsForm);
    }

    /**
     * @param FormInterface $labelsForm
     * @param ManagerRegistry $managerRegistry
     * @param Device $device
     * @return RedirectResponse
     */
    private function processCustomLabelsFormSubmit(
		FormInterface $labelsForm,
		ManagerRegistry $managerRegistry,
		Device $device
	): RedirectResponse
    {
        $formData = $labelsForm->getData();
        $deviceLabels = $formData['deviceLabels'];
        $customLabels = $formData['labels'];

        $deviceLabelsIds = [];

        foreach ($deviceLabels as $label) {

            $deviceLabelsIds[] = $label->getLabel()->getId();

        }

        foreach ($customLabels as $label) {

            //if not exists
            if (!in_array($label, $deviceLabelsIds)) {

                $label = $managerRegistry->getRepository(Label::class)->findOneBy(['id' => $label]);

                $labelDeviceMapping = new LabelDeviceMap();
                $labelDeviceMapping->setLabel($label);
                $labelDeviceMapping->setDevice($device);
                $managerRegistry->getManager()->persist($labelDeviceMapping);
                $managerRegistry->getManager()->flush();

            }

            //Remove processed labels from array
            $key = array_search($label, $deviceLabelsIds);
            if (false !== $key) {
                unset($deviceLabelsIds[$key]);
            }

        }

        foreach ($deviceLabelsIds as $labelDeleteId) {

            $label = $managerRegistry->getRepository(Label::class)->findOneBy(['id' => $labelDeleteId]);
            $device = $managerRegistry->getRepository(Device::class)->findOneBy(['id' => $device->getId()]);
            $labelDeviceMapping = $managerRegistry->getRepository(LabelDeviceMap::class)->findOneBy(
                ['label' => $label, 'device' => $device]);

            if ($labelDeviceMapping) {

                $managerRegistry->getManager()->remove($labelDeviceMapping);
                $managerRegistry->getManager()->flush();

            }

        }

        $this->addFlash(
            'success',
            'Labels successfully saved.'
        );

        return $this->redirectToRoute('admin_devices_detail', ['deviceId' => $device->getId()]);
    }

    /**
     * @param FormInterface $systemLabelsForm
     * @param ManagerRegistry $managerRegistry
     * @param Device $device
     * @return RedirectResponse
     */
    private function processSystemLabelsFormSubmit(FormInterface $systemLabelsForm, ManagerRegistry $managerRegistry, Device $device): RedirectResponse
    {
        $formData = $systemLabelsForm->getData();
        $deviceLabels = $formData['deviceLabels'];
        $systemLabels = $formData['labels'];

        $deviceLabelsIds = [];

        foreach ($deviceLabels as $label) {

            $deviceLabelsIds[] = $label->getSystemLabel()->getId();

        }

        foreach ($systemLabels as $label) {

            //if not exists
            if (!in_array($label, $deviceLabelsIds)) {

                $label = $managerRegistry->getRepository(SystemLabel::class)->findOneBy(['id' => $label]);

                $labelDeviceMapping = new SystemLabelDeviceMap();
                $labelDeviceMapping->setSystemLabel($label);
                $labelDeviceMapping->setDevice($device);
                $managerRegistry->getManager()->persist($labelDeviceMapping);
                $managerRegistry->getManager()->flush();

            }

            //Remove processed labels from array
            $key = array_search($label, $deviceLabelsIds);
            if (false !== $key) {
                unset($deviceLabelsIds[$key]);
            }

        }

        foreach ($deviceLabelsIds as $labelDeleteId) {

            $label = $managerRegistry->getRepository(SystemLabel::class)->findOneBy(['id' => $labelDeleteId]);
            $device = $managerRegistry->getRepository(Device::class)->findOneBy(['id' => $device->getId()]);
            $labelDeviceMapping = $managerRegistry->getRepository(SystemLabelDeviceMap::class)->findOneBy(
                ['systemLabel' => $label, 'device' => $device]);

            if ($labelDeviceMapping) {

                $managerRegistry->getManager()->remove($labelDeviceMapping);
                $managerRegistry->getManager()->flush();

            }

        }

        $this->addFlash(
            'success',
            'System Labels successfully saved.'
        );

        return $this->redirectToRoute('admin_devices_detail', ['deviceId' => $device->getId()]);
    }

    /**
     * @param FormInterface $netDevicesForm
     * @param Device $device
     * @param ManagerRegistry $managerRegistry
     * @return RedirectResponse
     */
    private function processNetDevicesFormSubmit(FormInterface $netDevicesForm, Device $device, ManagerRegistry $managerRegistry): RedirectResponse
    {
        $data = $netDevicesForm->getData();

        try {

            foreach ($device->getNetDevices() as $netDevice) {

                $formData = $data['cellular_' . $netDevice->getId()];
                if ($netDevice->isCellular !== $formData) {

                    $netDevice->setIsCellular($formData);
                    $managerRegistry->getManager()->persist($netDevice);
                    $managerRegistry->getManager()->flush();

                }

                $formData = $data['linkable_' . $netDevice->getId()];
                if ($netDevice->isLinkable !== $formData) {

                    $netDevice->setIsLinkable($formData);
                    $managerRegistry->getManager()->persist($netDevice);
                    $managerRegistry->getManager()->flush();

                }

            }

        } catch (\Exception $e) {

            $this->addFlash(
                'danger',
                'Network devices cannot be updated.'
            );

            return $this->redirectToRoute('admin_devices_detail', ['deviceId' => $device->getId()]);

        }

        $this->addFlash(
            'success',
            'Network devices settings was successfully updated.'
        );

        return $this->redirectToRoute('admin_devices_detail', ['deviceId' => $device->getId()]);
    }

    /**
     * @param FormInterface $monitoringGroupsForm
     * @param ManagerRegistry $managerRegistry
     * @param Device $device
     * @return RedirectResponse
     */
    private function processMonitoringGroupsFormSubmit(FormInterface $monitoringGroupsForm, ManagerRegistry $managerRegistry, Device $device): RedirectResponse
    {
        $formData = $monitoringGroupsForm->getData();
        $deviceMonitoringGroups = $formData['deviceMonitoringGroups'];
        $monitoringGroups = $formData['monitoringGroups'];

        $deviceMonitoringGroupsIds = [];

        foreach ($deviceMonitoringGroups as $monitoringGroup) {

            if ($monitoringGroup->getMonitoringGroup()->getId() === 1) {
                continue;
            }
            $deviceMonitoringGroupsIds[] = $monitoringGroup->getMonitoringGroup()->getId();

        }

        foreach ($monitoringGroups as $monitoringGroup) {

            //if not exists
            if (!in_array($monitoringGroup, $deviceMonitoringGroupsIds, true)) {

                $monitoringGroup = $managerRegistry->getRepository(MonitoringGroup::class)->findOneBy(['id' => $monitoringGroup]);

                $monitoringGroupDeviceMapping = new MonitoringGroupMapping();
                $monitoringGroupDeviceMapping->setMonitoringGroup($monitoringGroup);
                $monitoringGroupDeviceMapping->setDevice($device);
                $managerRegistry->getManager()->persist($monitoringGroupDeviceMapping);
                $managerRegistry->getManager()->flush();

            }

            //Remove processed labels from array
            $key = array_search($monitoringGroup, $deviceMonitoringGroupsIds);
            if (false !== $key) {
                unset($deviceMonitoringGroupsIds[$key]);
            }

        }

        foreach ($deviceMonitoringGroupsIds as $monitoringGroupDeleteId) {

            $monitoringGroup = $managerRegistry->getRepository(MonitoringGroup::class)->findOneBy(['id' => $monitoringGroupDeleteId]);
            $device = $managerRegistry->getRepository(Device::class)->findOneBy(['id' => $device->getId()]);
            $monitoringGroupDeviceMapping = $managerRegistry->getRepository(MonitoringGroupMapping::class)->findOneBy(
                ['monitoringGroup' => $monitoringGroup, 'device' => $device]);

            if ($monitoringGroupDeviceMapping) {

                $managerRegistry->getManager()->remove($monitoringGroupDeviceMapping);
                $managerRegistry->getManager()->flush();

            }

        }

        $this->addFlash(
            'success',
            'Monitoring groups successfully saved.'
        );

        return $this->redirectToRoute('admin_devices_detail', ['deviceId' => $device->getId()]);
    }

    /**
     * @param FormInterface $fwVersionForm
     * @param Device $device
     * @param ManagerRegistry $managerRegistry
     * @return RedirectResponse
     */
    private function processFwVersionFormSubmit(FormInterface $fwVersionForm, Device $device, ManagerRegistry $managerRegistry): RedirectResponse
    {
        // $form->getData() holds the submitted values
        // but, the original `$task` variable has also been updated
        $deviceFwForm = $fwVersionForm->getData();

        if ($fwVersionForm->offsetExists('no_specific') && $fwVersionForm->get('no_specific')->isClicked()) {

            $device->setOsVerDesired(null);
            $managerRegistry->getManager()->persist($device);
            $managerRegistry->getManager()->flush();

            $this->addFlash(
                'success',
                'Fw version was set to no specific.'
            );

            return $this->redirectToRoute('admin_devices_detail', ['deviceId' => $device->getId()]);

        }

        $osVersion = $managerRegistry->getRepository(DeviceOsVersion::class)
            ->findOneBy(['id' => $deviceFwForm['fw_version']]);

        $device->setOsVerDesired($osVersion);
        $managerRegistry->getManager()->persist($device);
        $managerRegistry->getManager()->flush();

        $this->addFlash(
            'success',
            'New Fw version was set.'
        );

        return $this->redirectToRoute('admin_devices_detail', ['deviceId' => $device->getId()]);
    }

	private function processConfigProfileFormSubmit(
		FormInterface $configProfileForm,
		Device $device,
		ManagerRegistry $managerRegistry,
		ConfigProfiles $configProfilesApi
	): RedirectResponse
	{

		$configProfileData = $configProfileForm->getData();

		$configProfile = $managerRegistry->getRepository(ConfigProfile::class)
			->findOneBy(['id' => $configProfileData['config_profile']]);

		if(!$configProfile){

			$this->addFlash(
				'danger',
				'Nonexisting Config Profile given.'
			);

			return $this->redirectToRoute('admin_devices_detail', ['deviceId' => $device->getId()]);

		}

		if ($configProfileForm->offsetExists('unset') && $configProfileForm->get('unset')->isClicked()) {

			try{

				$configProfilesApi->unassignDeviceFromConfigProfile([$device->getId()]);

			}catch (\Exception $exception){

				$this->addFlash(
					'danger',
					'There was an error.'
				);

			}

			$this->addFlash(
				'success',
				'Config Profile was successfully unassigned.'
			);

			return $this->redirectToRoute('admin_devices_detail', ['deviceId' => $device->getId()]);

		}

		try{

			$configProfilesApi->assignDeviceToConfigProfile($configProfile->getId(), [$device->getId()]);

			$this->addFlash(
				'success',
				'Config Profile was successfully assigned.'
			);

		}catch (\Exception $exception){

			$this->logger->error($exception->getMessage());

			$this->addFlash(
				'danger',
				'There was an error.'
			);
		}

		return $this->redirectToRoute('admin_devices_detail', ['deviceId' => $device->getId()]);

	}

	/**
	 * @param FormInterface $hostnameForm
	 * @param Device $device
	 * @param ManagerRegistry $managerRegistry
	 * @return RedirectResponse
	 */
	private function processHostnameFormSubmit(
		FormInterface $hostnameForm,
		Device $device,
		ManagerRegistry $managerRegistry
	): RedirectResponse
	{

		// $form->getData() holds the submitted values
		// but, the original `$task` variable has also been updated
		$formData = $hostnameForm->getData();

		$device->setHostnameDesired($formData['hostname']);
		$managerRegistry->getManager()->persist($device);
		$managerRegistry->getManager()->flush();

		$this->addFlash(
			'success',
			'Hostname will be updated.'
		);

		return $this->redirectToRoute('admin_devices_detail', ['deviceId' => $device->getId()]);

	}

	/**
	 * @param FormInterface $agentProfilesForm
	 * @param ManagerRegistry $managerRegistry
	 * @param AgentProfiles $agentProfilesApi
	 * @param Device $device
	 * @return RedirectResponse
	 */
	private function processAgentProfileFormSubmit(
		FormInterface $agentProfilesForm,
		ManagerRegistry $managerRegistry,
		AgentProfiles $agentProfilesApi,
		Device $device): RedirectResponse
	{
		$formData = $agentProfilesForm->getData();
		$agentProfileId = $formData['agent_profile'];

		$agentProfile = $managerRegistry->getRepository(AgentProfile::class)
			->findOneBy(['id' => $agentProfileId]);

		try{

			$agentProfilesApi->assignDeviceAgentProfile($agentProfileId, [$device->getId()]);

			$this->addFlash(
				'success',
				'Agent profile was updated.'
			);

		}catch (\Exception $exception){

			$this->logger->error($exception->getMessage());

			$this->addFlash(
				'danger',
				'Agent profile cannot be updated.'
			);

		}

		return $this->redirectToRoute('admin_devices_detail', ['deviceId' => $device->getId()]);
	}

	/**
	 * @param FormInterface $authProfilesForm
	 * @param ManagerRegistry $managerRegistry
	 * @param Device $device
	 * @return RedirectResponse
	 */
	private function processAuthProfileFormSubmit(
		FormInterface $authProfilesForm,
		ManagerRegistry $managerRegistry,
		Device $device,
		): RedirectResponse
	{
		$formData = $authProfilesForm->getData();
		$authProfileId = $formData['auth_profile'];

		$authProfile = $managerRegistry->getRepository(AuthProfile::class)
			->findOneBy(['id' => $authProfileId]);

		$device->setAuthProfile($authProfile);
		$managerRegistry->getManager()->persist($device);
		$managerRegistry->getManager()->flush();

		$this->addFlash(
			'success',
			'Auth profile was updated.'
		);

		return $this->redirectToRoute('admin_devices_detail', ['deviceId' => $device->getId()]);
	}

	/**
	 * @param FormInterface $customModuleDownloadForm
	 * @param Command $commandService
	 * @param ManagerRegistry $managerRegistry
	 * @param ParameterBagInterface $params
	 * @param Device $device
	 * @return RedirectResponse|BinaryFileResponse
	 */
	private function processCustomModuleDownloadFormSubmit(
		FormInterface $customModuleDownloadForm,
		Command $commandService,
		ManagerRegistry $managerRegistry,
		ParameterBagInterface $params,
		Device $device
	): RedirectResponse|BinaryFileResponse
	{
		$formData = $customModuleDownloadForm->getData();

		$devId = $formData['deviceId'];
		$deviceTypeId = $formData['deviceType'];

		$exitCode = $commandService->runCustomizedAgentGenerator($devId);

		if ($exitCode === 0) {

			$deviceType = $managerRegistry->getRepository(DeviceType::class)->findOneBy(['id' => $deviceTypeId]);

			$mapping = $deviceType->getDeviceTypeAgentMapping();
			if (count($mapping) > 1) {

				$this->addFlash(
					'danger',
					'No valid agent data given.'
				);

			} else {

				$customizedAgentPath = $params->get('app.customizedAgentPath');
				$agent = $mapping[0]->getDeviceAgent();

				$file = $customizedAgentPath . $devId . DIRECTORY_SEPARATOR . $agent->getDirName() . DIRECTORY_SEPARATOR . $agent->getPackageName();

				$response = new BinaryFileResponse($file);
				$response->setContentDisposition(
					ResponseHeaderBag::DISPOSITION_ATTACHMENT,
					$agent->getPackageName()
				);

				return $response;
			}

		} else {

			$this->addFlash(
				'danger',
				'The agent cannot be created.'
			);

		}

		return $this->redirectToRoute('admin_devices_detail', ['deviceId' => $device->getId()]);
	}

	/**
	 * @param ManagerRegistry $managerRegistry
	 * @param ApiService $apiService
	 * @param Device $device
	 * @return RedirectResponse
	 */
	private function processDeleteFormSubmit(
		ManagerRegistry $managerRegistry,
		ApiService $apiService,
		Device $device
	): RedirectResponse
	{
		$snmpDevice = $managerRegistry->getRepository(SnmpDevice::class);

		if ($apiService->deleteDevice($device->getId()) !== false) {

			try {
				$snmpDevice->removeSnmpDevice($device->getId());
				$this->addFlash(
					'success',
					'Device was successfully deleted.'
				);
			} catch (\Exception $e) {
				$this->addFlash(
					'danger',
					'Device was not deleted.'
				);
			}

		} else {

			$this->addFlash(
				'danger',
				'Device was not deleted.'
			);

		}

		return $this->redirectToRoute('admin_devices');
	}

}