<?php

namespace App\Form;

use Symfony\Component\Form\AbstractType;
use Symfony\Component\Form\Extension\Core\Type\CheckboxType;
use Symfony\Component\Form\Extension\Core\Type\ChoiceType;
use Symfony\Component\Form\Extension\Core\Type\FormType;
use Symfony\Component\Form\Extension\Core\Type\IntegerType;
use Symfony\Component\Form\Extension\Core\Type\PasswordType;
use Symfony\Component\Form\Extension\Core\Type\SubmitType;
use Symfony\Component\Form\Extension\Core\Type\TextType;
use Symfony\Component\Form\FormBuilderInterface;
use Symfony\Component\OptionsResolver\OptionsResolver;
use Symfony\Component\Validator\Constraints\Email;
use Symfony\Component\Validator\Constraints\NotBlank;

final class SettingsType extends AbstractType {

    /**
     * @param FormBuilderInterface $builder
     * @param array $options
     */
    public function buildForm(FormBuilderInterface $builder, array $options) {
        $isIpAddressEditEnabled = $options['data']['isIpAddressEditEnabled'];
        $isSlackSettingsComplete = $options['data']['isSlackSettingsComplete'];
        $isEmailSettingsComplete = $options['data']['isEmailSettingsComplete'];
        $configFormData = $options['data']['configFormData'];

        $builder
            ->add('serverIpAddress', TextType::class, [
                'block_prefix' => 'server',
                'disabled' => $isIpAddressEditEnabled,
                'label' => 'Server IP address or FQDN (e.g. my.server.com)',
                'constraints' => [
                    new NotBlank(),
                    // new Ip(),
                ],
            ])
            ->add('enableEmailNotifications', CheckboxType::class, [
                'label' => 'Enable Email Notifications',
                'required' => false,
            ])
            ->add('enableEmailNotificationToAddress', CheckboxType::class, [
                'label' => 'Send Email Notifications to other address:',
                'required' => false,
            ])
            ->add('notificationEmail', TextType::class, [
                'block_prefix' => 'notifications',
                'label' => 'Email address:',
                'required' => false,
                'constraints' => [
                    new Email(),
                ],
            ])
            ->add('senderEmail', TextType::class, [
                'block_prefix' => 'smtp',
                'label' => 'Sender Email address',
                'required' => false,
                'constraints' => [
                    new Email(),
                ],
            ])
            ->add('smtpServer', TextType::class, [
                'block_prefix' => 'smtp',
                'label' => 'SMTP Server name',
                'required' => false,
                'constraints' => [

                ],
            ])
            ->add('smtpPort', TextType::class, [
                'block_prefix' => 'smtp',
                'label' => 'SMTP Server port',
                'required' => false,
                'constraints' => [
                    new NotBlank(),
                ],
            ])
            ->add('smtpSecurity', ChoiceType::class, [
                'required' => true,
                'choices' => ['None' => 0, 'SSL/TLS' => 1, 'STARTTLS' => 2],
                'placeholder' => 'Select Security Type',
                'block_prefix' => 'smtp',
            ])
            ->add('smtpLogin', TextType::class, [
                'block_prefix' => 'smtp',
                'label' => 'SMTP Login name',
                'required' => false,
                'constraints' => [

                ],
            ])
            ->add('smtpPassword', PasswordType::class, [
                'block_prefix' => 'smtp',
                'label' => 'SMTP Login password',
                'required' => false,
                'constraints' => [

                ],
            ])
            ->add('slackToken', TextType::class, [
                'block_prefix' => 'notifications',
                'label' => 'Slack token',
                'required' => false,
                'constraints' => [

                ],
            ])
            ->add('slackChannel', TextType::class, [
                'block_prefix' => 'notifications',
                'label' => 'Slack channel name',
                'required' => false,
                'constraints' => [

                ],
            ])
            ->add('deviceTableColumns', ChoiceType::class, [
                'required' => false,
                'choices' => $options['data']['deviceTableColumnsChoices'],
                'multiple' => true,
                // 'mapped' => false,
            ])
            ->add('snmpGatewayThreads', IntegerType::class, [
                'block_prefix' => 'snmp',
                'label' => 'Threads',
                'required' => true,
                'constraints' => [
                    new NotBlank(),
                ],
            ])
            ->add('snmpGatewayBatch', IntegerType::class, [
                'block_prefix' => 'snmp',
                'label' => 'Batch',
                'required' => true,
                'constraints' => [
                    new NotBlank(),
                ],
            ])
            ->add('snmpGatewayTimeout', IntegerType::class, [
                'block_prefix' => 'snmp',
                'label' => 'Timeout',
                'required' => true,
                'constraints' => [
                    new NotBlank(),
                ],
            ])
            ->add('snmpGatewayRetries', IntegerType::class, [
                'block_prefix' => 'snmp',
                'label' => 'Gateway Retries',
                'required' => true,
                'constraints' => [
                    new NotBlank(),
                ],
            ])
            ->add('snmpGatewayInterval', IntegerType::class, [
                'block_prefix' => 'snmp',
                'label' => 'Interval',
                'required' => true,
                'constraints' => [
                    new NotBlank(),
                ],
            ])
			->add('snmpLogLevel', IntegerType::class, [
				'block_prefix' => 'snmp',
				'label' => 'Log Level',
				'required' => true,
				'constraints' => [
					new NotBlank(),
				],
			])
            ->add('snmpSslOverride', IntegerType::class, [
                'block_prefix' => 'snmp',
                'label' => 'Enforced SSL mode',
                'required' => true,
                'constraints' => [
                    new NotBlank(),
                ],
            ]);

        if (!empty($configFormData)) {
            $builder->add(
                $builder->create('config', FormType::class, ['inherit_data' => true]),
            );
            $groupBuilder = $builder->get('config');
            foreach ($configFormData as $configFormItem) {
                // Get just writable data, the rest is just for info
                if ($configFormItem['writable'] === true) {
                    $groupBuilder->add($configFormItem['name'] . '_enabled', CheckboxType::class, [
                        'label' => 'User Defined',
                        'label_attr' => ['class' => 'switch-custom'],
                        'required' => false,
                        'data' => strlen($configFormItem['value']) !== 0, // !empty($configFormItem['value']),//does not work, why???
                        // 'attr' => ['data-toggle' => 'toggle', 'data-size' => 'xs', 'data-on' => "User", 'data-off' => "System", 'data-offstyle'=>"success"],
                        // 'value' => true,

                        // 'disabled' => $configFormItem['disabled']
                    ]);

                    if ($configFormItem['type'] === 'bool') {
                        $groupBuilder->add($configFormItem['name'], CheckboxType::class, [
                            'label' => $configFormItem['label'] . ' (Actual value: <b>'
                                . (($configFormItem['actualValue'] === '1') ? 'true' : 'false') . '</b>)',
                            'label_html' => true,
                            'help' => $configFormItem['description'] . '.<br>The default value is: <b>'
                                . (($configFormItem['default'] === '0') ? 'false' : 'true')
                                . '</b>' . (($configFormItem['min'] === $configFormItem['max']) ? '.<b> The value cannot be changed!</b>' : '') . '.',
                            'help_html' => true,
                            'required' => false,
                            'data' => (bool)$configFormItem['value'],
                            // 'disabled' => $configFormItem['disabled'],
                        ]);
                    // dump($configFormItem);
                    } elseif ($configFormItem['type'] === 'int') {
                        // Integer value
                        $attr = [
                            'min' => $configFormItem['min'],
                            'max' => $configFormItem['max'],
                        ];
                        if (empty($configFormItem['value'])) {
                            // If not set disable component
                            $attr['readonly'] = 'readonly';
                        }

                        $groupBuilder->add($configFormItem['name'], IntegerType::class, [
                            'label' => $configFormItem['label'] . ' (Actual value: <b>'
                                . $configFormItem['actualValue'] . ' ' . $configFormItem['unit'] . '</b>)',
                            'label_html' => true,
                            'help' => $configFormItem['description'] . '.<br>The default value is: <b>'
                                . $configFormItem['default']
                                . ((!empty($configFormItem['unit'])) ? ' ' . $configFormItem['unit'] : '')
                                . '</b>. Value must be between: <b>' . $configFormItem['min'] . '</b> and <b>'
                                . $configFormItem['max']
                                . ((!empty($configFormItem['unit'])) ? ' ' . $configFormItem['unit'] : '') . '</b>.',
                            'help_html' => true,
                            // 'required' => true,
                            // 'disabled' => !$configFormItem['disabled'],
                            'data' => (!empty($configFormItem['value'])) ? $configFormItem['value'] : null,
                            'attr' => $attr,

                        ]);
                        // dump($configFormItem);
                    }
                }

                // dump($configFormItem);
            }
        }

        // die();

        $builder->add('save', SubmitType::class, ['label' => '<i class="fa-solid fa-floppy-disk" aria-hidden="true"></i> Save', 'label_html' => true]);

        if ($isSlackSettingsComplete) {
            $builder->add(
                'sendSlackTestNotification',
                SubmitType::class,
                ['label' => 'Send Slack Test Notification', 'attr' => ['class' => 'btn btn-xs btn-primary']],
            );
        }

        if ($isEmailSettingsComplete) {
            $builder->add(
                'sendEmailTestNotification',
                SubmitType::class,
                ['label' => 'Send Email Test Notification', 'attr' => ['class' => 'btn btn-xs btn-primary']],
            );
        }
    }

    public function configureOptions(OptionsResolver $resolver): void {
        $resolver->setDefaults(
            [
                // 'allow_extra_fields' => true
            ],
        );
    }

}
