<?php

declare(strict_types=1);

namespace App\Controller\Admin\Ajax;

use App\Entity\Snmp\Device;
use App\Form\Snmp\SnmpDeviceType;
use Doctrine\DBAL\Exception\UniqueConstraintViolationException;
use Doctrine\Persistence\ManagerRegistry;
use Symfony\Bundle\FrameworkBundle\Controller\AbstractController as Controller;
use Symfony\Component\HttpFoundation\JsonResponse;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\HttpFoundation\Response;
use Symfony\Component\Routing\Annotation\Route;

#[Route(path: '/admin/ajax/snmp', name: 'ajax_snmp_device_')]
class SnmpGatewayController extends Controller
{

    /**
     * Create a new Snmp Device
     *
     * @param Request $request
     * @param ManagerRegistry $managerRegistry
     * @return Response
     */
    #[Route(path: '/create', name: 'create', methods: ['POST'])]
    public function createSnmpEntryAction(Request $request, ManagerRegistry $managerRegistry): Response {

        $snmpDevice = new Device();
        $form = $this->createForm(SnmpDeviceType::class, ['snmpDevice' => $snmpDevice]);
        $form->handleRequest($request);

        if ($form->isSubmitted()) {

            $formData = $form->getData();
            $em = $managerRegistry->getManager();

            $rawFormData = $request->getPayload()->all();

            try {

                $snmpDevice->setAddress($formData['address']);
                $snmpDevice->setCommunityString($formData['communityString']);
                $snmpDevice->setProtocolVersion((int) $rawFormData['snmp_device']['protocolVersion']);
                $em->persist($snmpDevice);
                $em->flush();

            }catch (UniqueConstraintViolationException){

                return new JsonResponse(['code' => 400, 'message' => 'error', 'errors' => ['error' => "Device already exists."]],
                    400);

            }catch (\Exception $exception){

                //dump($exception->getMessage());

                return new JsonResponse(['code' => 400, 'message' => 'error', 'errors' => ['error' => "Unable to create the Snmp Device."]],
                    400);

            }

            return new JsonResponse(['code' => 200, 'message' => 'Device successfully created.', 'success' => ''],
                200);
        }

        return new Response('Bad request.', 400);
    }

    #[Route(path: '/delete/{deviceId}', name: 'delete', methods: ['GET'])]
    public function deviceSnmpEntryAction(Request $request, int $deviceId, ManagerRegistry $managerRegistry): Response {


        if ($request->isXmlHttpRequest()) {

            $device = $managerRegistry->getRepository(Device::class)->findOneBy(['id' => $deviceId]);

            if($device){

                try {
                    $em = $managerRegistry->getManager();
                    $em->remove($device);
                    $em->flush();

                }catch (\Exception $exception){

                    return new JsonResponse(['code' => 400, 'message' => 'Device cannot be deleted: ',
                        'errors' => ['Something bad happened.']],
                        200);

                }

                return new JsonResponse(['code' => 200, 'message' => 'Device "'. $device->getAddress() .'" was deleted.', 'success' => ''],
                    200);

            }

        }

        return new Response('Bad request.', 400);

    }

}