<?php

namespace App\Controller\Admin;

use App\Repository\AlertRepository;
use App\Repository\ConfigRepository;
use App\Repository\DeviceAgentRepository;
use App\Repository\DeviceRepository;
use App\Repository\MonitoringGroupRepository;
use App\Service\Api\AgentProfiles;
use App\Service\Api\ApiService;
use App\Service\Command;
use App\Service\ConfigurationService;
use Doctrine\DBAL\Exception;
use Symfony\Bundle\FrameworkBundle\Controller\AbstractController as Controller;
use Symfony\Component\DependencyInjection\ParameterBag\ParameterBagInterface;
use Symfony\Component\HttpFoundation\BinaryFileResponse;
use Symfony\Component\HttpFoundation\RedirectResponse;
use Symfony\Component\HttpFoundation\Response;
use Symfony\Component\Process\Exception\ProcessFailedException;
use Symfony\Component\Routing\Annotation\Route;

#[Route('/admin', name: 'admin_')]
class DashboardController extends Controller
{
    /**
     * @param DeviceAgentRepository $deviceAgentRepository
     * @param MonitoringGroupRepository $monitoringGroupRepository
     * @param AlertRepository $alertRepository
     * @param ApiService $apiService
     * @param DeviceRepository $deviceRepository
     */
    public function __construct(
        private readonly DeviceAgentRepository $deviceAgentRepository,
        private readonly MonitoringGroupRepository $monitoringGroupRepository,
        private readonly AlertRepository $alertRepository,
        private readonly ApiService $apiService,
        private readonly DeviceRepository $deviceRepository
    ) {
    }

    /**
     * @param ConfigurationService $configurationService
     * @param ConfigRepository $configRepository
     * @param Command $commandService
     * @param AgentProfiles $apiAgentProfiles
     * @return Response
     * @throws Exception
     */
    #[Route('/', name: 'dashboard')]
    public function default(
        ConfigurationService $configurationService,
        ConfigRepository $configRepository,
        Command $commandService,
        AgentProfiles $apiAgentProfiles
    ): Response {

        //$agentProfileResult = $apiAgentProfiles->createAgentProfile();
        //$agentProfileId = $agentProfileResult->operation_result->Code;

       /* dump($apiAgentProfiles->getAgentProfile($agentProfileId));

        dump($agentProfileId);

        die();*/

        if ($configurationService->isFirstRun()) {
            //return $this->redirectToRoute('admin_help', ['page' => 'welcome']);

            //$isFirstRun = $configurationService->isFirstRun();
            $configurationService->updateIsFirstRun(false);

            return $this->render("help/welcome.html.twig");
        }

        $lights = [];
        $monitoringGroups = [];
        $helptxt = '';
        $triggeredAlerts = [];
        $triggeredDevices = [];
        $invalidAlerts = [];
        $pastAlerts = [];
        $pastDevices = [];

        $data = $this->apiService->getDashboardData();

        //dump($data);
        //die();

        if ($data) {
            $lights = $this->getDashboardLightsData($data->lights);
            $helptxt = $data->helptxt;
            $monitoringGroups = $this->getDashboardMonitoringGroupsData($data->groups);
            [$triggeredAlerts, $triggeredDevices] = $this->getDashboardTriggeredAlertsData($data->{'triggered alerts'});
            $invalidAlerts = $this->getDashboardInvalidAlertsData($data->{'invalid alerts'});
            [$pastAlerts, $pastDevices] = $this->getDashboardPastAlertsData($data->{'past alerts'});
        } else {
            //No dashboard data given
            $caption = 'Some problem occurred.';
            $serviceStatus = $commandService->getServiceStatus();

            if ($serviceStatus === Command::SERVICE_STATUS_INACTIVE) {
                $caption = 'Service is down';
            }

            try {
                $processDumpLicense = $commandService->checkLicense();
                $processDumpLicense->mustRun();
                $licenseCheckExitCode = $processDumpLicense->getExitCode();
                if ($licenseCheckExitCode !== 0) {
                    $caption = 'License is invalid';
                }
            } catch (ProcessFailedException) {
                $caption = 'Cannot get the license info.';
            }

            $lights[] = [
                'name' => 'Global',
                'caption' => $caption,
                'state' => 'B',
                'onlight' => '',
            ];
        }

        return $this->render("dashboard/dashboard.html.twig", [
            'lights' => $lights,
            'help' => $helptxt,
            'groups' => $monitoringGroups,
            'triggeredAlerts' => $triggeredAlerts,
            'triggeredDevices' => $triggeredDevices,
            'pastAlerts' => $pastAlerts,
            'pastDevices' => $pastDevices,
            'invalidAlerts' => $invalidAlerts,
            'data' => $data,
        ]);
    }

    /**
     * @param array $aLights
     * @return array
     */
    private function getDashboardLightsData(array $aLights): array
    {

        $lights = [];

        foreach ($aLights as $light) {
            $lights[] = [
                'name' => $light->name,
                'caption' => $light->caption,
                'state' => $light->state,
                'onlight' => $light->onlight,
            ];
        }

        return $lights;
    }

    private function getDashboardMonitoringGroupsData(array $aMonitoringGroups): array
    {

        $groups = [];

        foreach ($aMonitoringGroups as $group) {
            $monitoringGroup = $this->monitoringGroupRepository->findOneBy(['id' => $group->id]);
            $groups[] = [

                'id' => $group->id,
                'name' => (isset($monitoringGroup)) ? $monitoringGroup->name : '',
                'state' => $group->state,
                'devices' => $group->devices,
                'lights' => $group->lights,

            ];
        }

        return $groups;
    }

    /**
     * @param array $aTriggeredAlerts
     * @return array
     * @throws Exception
     */
    private function getDashboardTriggeredAlertsData(array $aTriggeredAlerts): array
    {

        $triggeredAlerts = [];
        $triggeredDevices = [];

        foreach ($aTriggeredAlerts as $alertId) {
            $oAlert = $this->alertRepository->findOneBy(['id' => $alertId]);

            $triggeredAlerts[] = [
                'id' => $alertId,
                'name' => $oAlert->name,
                'targetMonitoringGroup' => $oAlert->targetMonitoringGroup,
            ];

            if (!empty($oAlert->getTargetDevice())) {
                //Target Device
                $triggeredDevices[$alertId] = [
                    'device' => $oAlert->getTargetDevice(),
                ];
            } elseif (!empty($oAlert->getTargetMonitoringGroup())) {
                //Target Monitoring group
                $devicesId = $this->apiService->getTriggeredAlertDevices($alertId);
                $devices = $this->deviceRepository->getByDevIds((array)$devicesId);

                $triggeredDevices[$alertId] = [

                    'devicesId' => $devicesId,
                    'device' => $devices,
                ];
            }
        }

        return [$triggeredAlerts, $triggeredDevices];
    }

    /**
     * @param array $aInvalidAlerts
     * @return array
     */
    private function getDashboardInvalidAlertsData(array $aInvalidAlerts): array
    {

        $invalidAlerts = [];

        foreach ($aInvalidAlerts as $alertId) {
            $oAlert = $this->alertRepository->findOneBy(['id' => $alertId]);
            $invalidAlerts[] = [
                'id' => $alertId,
                'name' => $oAlert->name,
            ];
        }

        return $invalidAlerts;
    }

    /**
     * @return array
     * @throws Exception
     */
    private function getDashboardPastAlertsData(array $aPastAlerts): array
    {

        $pastAlerts = [];
        $pastDevices = [];

        foreach ($aPastAlerts as $alertId) {
            $oAlert = $this->alertRepository->findOneBy(['id' => $alertId]);

            $pastAlerts[] = [
                'id' => $alertId,
                'name' => $oAlert->name,
                'targetMonitoringGroup' => $oAlert->targetMonitoringGroup,
            ];

            if (!empty($oAlert->getTargetDevice())) {
                //Target Device
                $pastDevices[$alertId] = [
                    'device' => $oAlert->getTargetDevice(),
                ];
            } elseif (!empty($oAlert->getTargetMonitoringGroup())) {
                //Target Monitoring group
                $devicesId = $this->apiService->getPastAlertDevices($alertId);
                $devices = $this->deviceRepository->getByDevIds((array)$devicesId);

                $pastDevices[$alertId] = [
                    'devicesId' => $devicesId,
                    'device' => $devices,
                ];
            }
        }

        return [$pastAlerts, $pastDevices];
    }

    /**
     * Twig header global status
     * @return string
     */
    public function getColor(): string
    {

        $data = $this->apiService->getDashboardData();

        if ($data && is_array($data->lights)) {
            foreach ($data->lights as $light) {
                if ($light->name === 'Global') {
                    return $light->state;
                }
            }
        }

        return 'grey';
    }

    /**
     * @return Response
     * @throws \Exception
     */
    #[Route(path: '/support', name: 'support_redirect')]
    public function getSupportRedirect(ConfigurationService $configurationService): Response
    {

        if (empty($configurationService->getSupportLink())) {
            return $this->redirectToRoute('admin_help', ['page' => 'compendium']);
            ;
        }
        return $this->redirect($configurationService->getSupportLink());
    }

    /**
     * Twig get download on header
     * @return \App\Entity\DeviceAgent[]
     */
    public function getDownloads(): array
    {

        return $this->deviceAgentRepository->findBy(['showInShortcut' => true]);
    }

    /**
     *
     * @return BinaryFileResponse
     */
    #[Route(path: '/rss', name: 'rss')]
    public function getRss(ParameterBagInterface $parameterBag): BinaryFileResponse
    {

        $publicPath = $parameterBag->get('kernel.project_dir') .  '/public/';
        ;
        $filename = "mevents.rss";

        // This should return the file located in /mySymfonyProject/web/public-resources/TextFile.txt
        // to being viewed in the Browser
        return new BinaryFileResponse($publicPath . $filename);
    }
}
