<?php

declare(strict_types=1);

namespace App\Controller\Admin\Ajax;

use App\Entity\AgentProfile\AgentProfile;
use App\Entity\ConfigProfile\ConfigProfile;
use App\Entity\Device;
use App\Entity\DeviceOsVersion;
use App\Repository\DeviceRepository;
use App\Service\Api\AgentProfiles;
use Doctrine\DBAL\ArrayParameterType;
use Doctrine\DBAL\Connection;
use Doctrine\DBAL\Exception;
use Doctrine\DBAL\ParameterType;
use Doctrine\Persistence\ManagerRegistry;
use Symfony\Bundle\FrameworkBundle\Controller\AbstractController as Controller;
use Symfony\Component\HttpFoundation\JsonResponse;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\HttpFoundation\Response;
use Symfony\Component\Routing\Annotation\Route;

#[Route(path: '/admin/ajax/device', name: 'ajax_device_')]
class DeviceController extends Controller
{

    /**
     * Get devices not assigned to web tunnels
     * @return JsonResponse|Response
     * @throws Exception
     */
    #[Route(path: '/not-assigned-web-tunnel', name: 'get_not_web_tunnel', methods: ['GET'])]
    public function getAllDevices(Request $request, DeviceRepository $deviceRepository): Response {

        if ($request->isXmlHttpRequest()) {

            $devices = $deviceRepository->findAllNotAssignedToWebTunnelPlainSql();
            $result = [];

            foreach ($devices as $device) {
                $result[$device['id_device']] = ['value' => $device['devid'], 'description' => $device['name']];
            }

            return new JsonResponse($result);
        }

        return new Response('Bad request.', 400);

    }

    /**
     * Get all Devices not in Monitoring Group.
     * @return Response
     */
    #[Route(path: '/get/all-not/{mntGroupId}', name: 'get_all_not_in_monitoring_group', methods: ['GET'])]
    public function getAllDevicesNotInMonitoringGroup(Request          $request,
                                                      DeviceRepository $deviceRepository,
                                                      int $mntGroupId): Response {

        if ($request->isXmlHttpRequest()) {

            $devices = $deviceRepository->findAllNotInMonitoringGroupPlainSql($mntGroupId);

            $result = [];

            foreach ($devices as $device) {
                $result[$device['id_device']] = ['value' => $device['devid'], 'description' => $device['name']];
            }

            return new JsonResponse($result);
        }

        return new Response('Bad request.', 400);

    }

    /**
     * Add Devices to Monitoring Groups
     * @param Request $request
     * @param ManagerRegistry $managerRegistry
     * @return Response
     */
    #[Route(path: '/add/monitoring-group', name: 'add_to_monitoring_group', methods: ['POST'])]
    public function addDevicesToMonitoringGroupAction(Request $request, ManagerRegistry $managerRegistry): Response
    {

        if (!$request->isXmlHttpRequest()) {

            return new Response('Bad request.', 400);

        }

        if (isset($request->request)) {

            $monitoringGroups = (array)$request->get('monitoringGroups');
            $selectedDevices = $request->get('selectedDevices');

            if (!isset($monitoringGroups) || !isset($selectedDevices))
                return new JsonResponse(['status' => 'Error'], 400);

            $deviceIds = [];

            foreach ($selectedDevices as $device) {
                $deviceIds[] = $device['id'];
            }

            /** @var Connection $connection */
            $connection =  $managerRegistry->getConnection();
            $connection->beginTransaction();

            foreach ($monitoringGroups as $monitoringGroupId) {

                $placeholders = [];
                $values = [];
                $types = [];

                foreach ($deviceIds as $deviceId) {

                    $placeholders[] = '(?)';
                    $values[] = [$monitoringGroupId, $deviceId];
                    $types[] = ArrayParameterType::INTEGER;

                }

                try{
                    $connection->executeStatement(
                        'INSERT IGNORE `monitoring_group_device_map` (`mongrp_id`, `id_device`)  VALUES ' .
                        implode(', ', $placeholders),
                        $values,
                        $types
                    );
                }catch (\Exception $e){

                    $connection->rollBack();

                    return new JsonResponse(['status' => 'Error', 'message' => 'Something bad happened.'],
                        400);

                }

            }

            $connection->commit();

            return new JsonResponse(['code' => 200, 'message' => 'Devices were successfully assigned to Monitoring Groups.', 'errors' => ['errors' => ['']]],
                200);

        }

        // If we reach this point, it means that something went wrong
        return new JsonResponse(['status' => 'Error', 'message' => 'Something bad happened.'],
            400);

    }

    /**
     * Add Labels to Devices
     * @param Request $request
     * @param ManagerRegistry $managerRegistry
     * @return Response
     */
    #[Route(path: '/add/label', name: 'add_label', methods: ['POST'])]
    public function addLabelToDeviceAction(Request $request, ManagerRegistry $managerRegistry): Response {

        if (!$request->isXmlHttpRequest()) {

            return new Response('Bad request.', 400);

        }

        if (isset($request->request)) {

            $labelIds = (array)$request->get('labels');
            $selectedDevices = $request->get('selectedDevices');

            if (!isset($labelIds) || !isset($selectedDevices))
                return new JsonResponse(['status' => 'Error'], 400);

            $deviceIds = [];

            foreach ($selectedDevices as $device) {
                $deviceIds[] = $device['id'];
            }

            //$labels = $managerRegistry->getRepository(Label::class)->findBy(['id' => $labelIds]);
            //$deviceRepository = $managerRegistry->getRepository(Device::class);

            $placeholders = [];
            $values = [];
            $types = [];

            foreach ($deviceIds as $deviceId) {

                foreach ($labelIds as $labelId) {
                    $placeholders[] = '(?)';
                    $values[] = [$labelId, $deviceId];
                    $types[] = ArrayParameterType::INTEGER;
                }

                /*$device = $deviceRepository->findOneBy(['id' => $device['id']]);

                foreach ($labels as $label) {

                    $existingLabel = $managerRegistry->getRepository(LabelDeviceMap::class)
                        ->findOneBy(['device' => $device, 'label' => $label]);

                    if(!$existingLabel){

                        try {

                            $labelDeviceMapping = new LabelDeviceMap();
                            $labelDeviceMapping->setLabel($label);
                            $labelDeviceMapping->setDevice($device);
                            $managerRegistry->getManager()->persist($labelDeviceMapping);
                            $managerRegistry->getManager()->flush();

                        } catch (UniqueConstraintViolationException) {

                            //Do nothing, label already assigned

                        } catch (\Exception) {

                        }

                    }

                }*/

            }

            /** @var Connection $connection */
            $connection =  $managerRegistry->getConnection();

            try{

                $connection->executeStatement(
                    'INSERT IGNORE `labels_map` (`id_label`, `id_device`)  VALUES ' .
                    implode(', ', $placeholders),
                    $values,
                    $types
                );

            }catch (\Exception $e){

                return new JsonResponse(['status' => 'Error', 'message' => 'Something bad happened.'],
                    400);

            }

            return new JsonResponse(['code' => "200", 'message' => 'Label was successfully assigned to the devices.', 'errors' => ['errors' => ['']]],
                200);

        }

        // If we reach this point, it means that something went wrong
        return new JsonResponse(['status' => 'Error', 'message' => 'Something bad happened.'],
            400);

    }

    /**
     * Add Fsw to Devices
     * @param Request $request
     * @param ManagerRegistry $managerRegistry
     * @return Response
     */
    #[Route(path: '/add/fw', name: 'add_fw', methods: ['POST'])]
    public function addFwToDeviceAction(Request $request, ManagerRegistry $managerRegistry): Response {

        if (!$request->isXmlHttpRequest()) {

            return new Response('Bad request.', 400);

        }

        if (isset($request->request)) {

            $fws = (array)$request->get('fws');
            $selectedDevices = $request->get('selectedDevices');

            if (!isset($fws) || !isset($selectedDevices))
                return new JsonResponse(['status' => 'Error'], 400);

            $fws = $managerRegistry->getRepository(DeviceOsVersion::class)->findBy(['id' => $fws]);
            $deviceRepository = $managerRegistry->getRepository(Device::class);

            $errors = [];
            $success = [];
            $values = [];

            foreach ($selectedDevices as $device) {

                $device = $deviceRepository->findOneBy(['id' => $device['id']]);

                $deviceType = $device->getDevType();

                foreach ($fws as $fw) {

                    $fwDeviceOsType = $fw->getOsType()->getDevType();

                    //Check if there is the same OS type on device as selected FW
                    if ($deviceType === $fwDeviceOsType) {

                        $values[] = $device->getId();
                        $success[] = $device->getDevId();

                    } else {

                        $errors[] = $device->getDevId();

                    }

                }

            }

            /** @var Connection $connection */
            $connection =  $managerRegistry->getConnection();

            try{

                $connection->beginTransaction();

                $globalVersion = $connection->executeQuery('SELECT `version` FROM `product_flags` WHERE `id` = "user_devchg_version"')->fetchOne();
                $connection->executeQuery('UPDATE `product_flags` SET `version` = `version` + 1 WHERE `id` = "user_devchg_version"')->fetchOne();
                $connection->executeStatement('UPDATE `device_stats` SET `sync` = 2 WHERE `id_device` IN (?)', [$values], [ArrayParameterType::INTEGER]);

                $connection->executeStatement(
                    'UPDATE `devices` SET `os_ver_desired` = ?, `gui_version` = ?  WHERE `id_device` IN (?)',
                    [$fw->getId(), $globalVersion + 1,  $values],
                    [ParameterType::INTEGER, ParameterType::INTEGER, ArrayParameterType::INTEGER]
                );

                $connection->commit();

            }catch (\Exception $e){

                $connection->rollback();

                return new JsonResponse(['status' => 'Error', 'message' => 'Something bad happened.'],
                    400);
            }

            return new JsonResponse([
                'code' => 200,
                'message' => 'Fw version successfully assigned to the devices: ',
                'success' => $success,
                'errors' => $errors
            ],
                200);

        }

        // If we reach this point, it means that something went wrong
        return new JsonResponse(['status' => 'Error', 'message' => 'Something bad happened.'],
            400);

    }

    /**
     * Add Agent Profile to Devices
     * @return Response
     */
    #[Route(path: '/add/agent-profile', name: 'add_agent_profile', methods: ['POST'])]
    public function addAgentProfileToDeviceAction(Request $request, ManagerRegistry $managerRegistry,
                                                  AgentProfiles $agentProfilesApi): Response {

        if (!$request->isXmlHttpRequest()) {

            return new Response('Bad request.', 400);

        }

        if (isset($request->request)) {

            $agentProfiles = (array)$request->get('agentProfiles');
            $selectedDevices = $request->get('selectedDevices');

            if (!isset($agentProfiles) && !isset($selectedDevices))
                return new JsonResponse(['status' => 'Error'], 400);

            $agentProfiles = $managerRegistry->getRepository(AgentProfile::class)->findBy(['id' => $agentProfiles]);
            $deviceRepository = $managerRegistry->getRepository(Device::class);

            $errors = [];
            $success = [];

            foreach ($agentProfiles as $agentProfile) {

                $deviceIds = [];
                $agentProfileId = $agentProfile->getId();

                foreach ($selectedDevices as $device) {

                    $device = $deviceRepository->findOneBy(['id' => $device['id']]);
                    $deviceIds[] = $device->getId();

                }

                try {

                    $result = $agentProfilesApi->assignDeviceAgentProfile($agentProfileId, $deviceIds);
                    if($result->operation_result->Code < 0){
                        throw new \Exception('Cannot call api');
                    }
                    /*$device->setAgentProfile($agentProfile);
                    $managerRegistry->getManager()->persist($device);
                    $managerRegistry->getManager()->flush();*/

                    $success[] = 'Selected profile was assigned.'; //$device->getDevId();

                } catch (\Exception $e) {

                    $errors[] = 'Something bad happend.' . $e->getMessage(); //$device->getDevId();

                }

            }

            return new JsonResponse([
                'code' => 200,
                'message' => 'Agent Profile was successfully assigned to the devices: ',
                'success' => $success,
                'errors' => $errors
            ],
                200);

        }

        // If we reach this point, it means that something went wrong
        return new JsonResponse(['status' => 'Error', 'message' => 'Something bad happened.'],
            400);

    }

    /**
     * Get Devices with Config Profile
     * @return Response
     * @throws Exception
     */
    #[Route(path: '/get/config-profiles', name: 'get_config_profiles', methods: ['GET'])]
    public function getConfigProfilesAction(Request $request, DeviceRepository $deviceRepository, ManagerRegistry $managerRegistry): Response
    {

        if ($request->isXmlHttpRequest()) {

            $devices = $deviceRepository->getDevicesToClonedFrom();

            $result = [];

            foreach ($devices as $device) {
                $result[$device['id_device']] = ['value' => $device['devid'], 'description' => $device['name']];
            }

            return new JsonResponse($result);
        }

        return new JsonResponse(['status' => 'Error', 'message' => 'Something bad happened.'],
            400);

    }

    /**
     * Get All Agent Profiles
     * @return Response
     * @throws Exception
     */
    #[Route(path: '/get/agent-profiles', name: 'get_agent_profiles', methods: ['GET'])]
    public function getAgentProfilesAction(Request $request, ManagerRegistry $managerRegistry): Response
    {

        if ($request->isXmlHttpRequest()) {

            $agentProfiles = $managerRegistry->getRepository(AgentProfile::class)->findAll();

            $result = [];

            foreach ($agentProfiles as $agentProfile) {
                $result[$agentProfile->getId()] = ['value' => $agentProfile->getProfileName(), 'description' => ''];
            }

            return new JsonResponse($result);
        }

        return new JsonResponse(['status' => 'Error', 'message' => 'Something bad happened.'],
            400);

    }

    /**
     * Add Config Profile to Devices
     * @return Response
     */
    #[Route(path: '/add/config-profile', name: 'add_config_profile', methods: ['POST'])]
    public function addConfigProfileToDeviceAction(Request $request, ManagerRegistry $managerRegistry): Response {

        if (!$request->isXmlHttpRequest()) {

            return new Response('Bad request.', 400);

        }

        if (isset($request->request)) {

            $devices = (array)$request->get('devices');
            $selectedDevices = $request->get('selectedDevices');
            $overwrite = $request->get('overwrite'); //retype to bool did not work for false

            if (!isset($devices) && !isset($selectedDevices))
                return new JsonResponse(['status' => 'Error'], 400);

            $devices = $managerRegistry->getRepository(Device::class)->findBy(['id' => $devices]);
            $deviceRepository = $managerRegistry->getRepository(Device::class);

            $errors = [];
            $success = [];

            foreach ($selectedDevices as $device) {

                $device = $deviceRepository->findOneBy(['id' => $device['id']]);

                $deviceType = $device->getDevType();

                foreach ($devices as $deviceCloneFrom) {

                    $deviceTypeCloneFrom = $deviceCloneFrom->getDevType();
                    $deviceConfigProfileFrom = $deviceCloneFrom->getConfigProfileAutomatic();

                    //Check if there is the same OS type on device as selected FW
                    if (($deviceType === $deviceTypeCloneFrom && $device !== $deviceCloneFrom) && (
                        ($device->getConfigProfile() instanceof ConfigProfile && $overwrite == 'true') ||
                        ($device->getConfigProfile() === null)) ) {

                        $values[] = $device->getId();
                        $success[] = $device->getDevId();

                    } else {

                        $errors[] = $device->getDevId();

                    }

                }

            }

            /** @var Connection $connection */
            $connection =  $managerRegistry->getConnection();

            try{

                $connection->beginTransaction();

                $globalVersion = $connection->executeQuery('SELECT `version` FROM `product_flags` WHERE `id` = "user_devchg_version"')->fetchOne();
                $connection->executeQuery('UPDATE `product_flags` SET `version` = `version` + 1 WHERE `id` = "user_devchg_version"')->fetchOne();
                $connection->executeStatement('UPDATE `device_stats` SET `sync` = 2 WHERE `id_device` IN (?)', [$values], [ArrayParameterType::INTEGER]);

                $connection->executeStatement(
                    'UPDATE `devices` SET `config_profile` = ?, `gui_version` = ?  WHERE `id_device` IN (?)',
                    [$deviceConfigProfileFrom->getId(), $globalVersion + 1,  $values],
                    [ParameterType::INTEGER, ParameterType::INTEGER, ArrayParameterType::INTEGER]
                );

                $connection->commit();

            }catch (\Exception $e){

                $connection->rollback();

                return new JsonResponse(['status' => 'Error', 'message' => $e->getMessage()],
                    400);
            }

            return new JsonResponse([
                'code' => 200,
                'message' => 'Config Profile was successfully assigned to the devices: ',
                'success' => $success,
                'errors' => $errors
            ],
                200);

        }

        // If we reach this point, it means that something went wrong
        return new JsonResponse(['status' => 'Error', 'message' => 'Something bad happened.'],
            400);

    }

    /**
     * Remove Config Profile for Devices
     * @return Response
     */
    #[Route(path: '/remove/config-profile', name: 'remove_config_profile', methods: ['POST'])]
    public function removeConfigProfileDeviceAction(Request $request, ManagerRegistry $managerRegistry,
                                                    DeviceRepository $deviceRepository): Response
    {

        if (!$request->isXmlHttpRequest()) {

            return new Response('Bad request.', 400);

        }

        if (isset($request->request)) {

            $selectedDevices = $request->get('selectedDevices');

            if (!isset($selectedDevices))
                return new JsonResponse(['status' => 'Error'], 400);

            $errors = [];
            $success = [];

            $deviceIds = [];

            foreach ($selectedDevices as $device) {
                $deviceIds[] = $device['id'];
            }

            $connection =  $managerRegistry->getConnection();

            try{

                $connection->beginTransaction();

                $globalVersion = $connection->executeQuery('SELECT `version` FROM `product_flags` WHERE `id` = "user_devchg_version"')->fetchOne();
                $connection->executeQuery('UPDATE `product_flags` SET `version` = `version` + 1 WHERE `id` = "user_devchg_version"')->fetchOne();
                $connection->executeStatement('UPDATE `device_stats` SET `sync` = 2 WHERE `id_device` IN (?)', [$deviceIds], [ArrayParameterType::INTEGER]);


                $connection->executeStatement(
                    'UPDATE `devices` SET `config_profile` = ?, `gui_version` = ?  WHERE `id_device` IN (?)',
                    [null, $globalVersion + 1,  $deviceIds],
                    [ParameterType::INTEGER, ParameterType::INTEGER, ArrayParameterType::INTEGER]
                );

                $connection->commit();

            }catch (\Exception $e){

                $connection->rollback();

                return new JsonResponse(['status' => 'Error', 'message' => 'Something bad happened.'],
                    400);
            }

            return new JsonResponse([
                'code' => 200,
                'message' => 'Config Profile was successfully removed for selected devices.',
                'success' => $success,
                'errors' => $errors
            ],
                200);

        }
    }

}