#!/bin/bash
# not supported by bash on router: shopt -s extglob

# Knobs... these may be interesting to mess with
g_logclean_days=7 # how many days to allow old PID directories to be left on filesystem
g_ctime_childexitwait=4 # how much time to give child agent to exit normally before killing it by -9 when monitor is exiting (keep this number small)
g_err_print_tolerance=0 # after how much uncounted errors the settings is stored (and thus counter propagated to UI); increase to make writes to disk less frequent on errors but also keep single errors invisible to UI
g_kill_output="yes" # stop output after initialization by default

# Constants ... these may not be wise to mess with
g_errc_inval=1 # EINVAL
g_errc_srvinval=2 # Invalid (NaN) response from server
g_errc_appinval=3 # Invalid (NaN) response from application
g_errc_srv=4 # Error code from server received
g_errc_app=5 # Error code from application received
g_errc_comm=6 # Error during communication with server (CURL error)
g_errc_rereg=200 # Internal-only error code to load settings after reregistration happened
g_txtll_debug="DEBUG:" # debug
g_txtll_info="INFO:" # info
g_txtll_note="LOG:" # log (notice)
g_txtll_warn="WARN:" # warning
g_txtll_err="ERR:" # error
g_txtll_fatal="FATAL:" # fatal
g_retcode_upgrade=2
g_txtappname_short="adwarfg"
g_txtappname_long="dwarfg-Shell-Agent"
g_appversion="1.0.1" # NOTE: must match agent_version in globals.cpp for release
g_protoversion="2.0"
g_sys_icros="Advantech router"
g_sys_linux="Linux box"
g_sys_owrt="OpenWRT box"
g_sys_teltonika="Teltonika router"
g_sys_cstech="CS-Tech device"
g_sys_other="Other system"
# Textual constants - do not touch these
# non-used characters: E H I J K L M N Q W
g_regerr_err="EEEE"
g_regerr_noop="NNNN"
g_regerr_devt="JJJJ"
g_regerr_license="LLLL"
g_regerr_comm="KKKK"
g_regerr_servid="IIII"
g_regerr_machid="MMMM"
g_fback_fwfnrouter=""
g_fback_fwfnlinux=""
g_deployd_systemd=1
g_deployd_sysv=2
g_deployd_own=3
g_ctxt_devidsep=":"
g_ctxt_fullvardump="FULLDUMP"
g_form_devtype="device_type"
g_form_devtoken="device_token"
g_form_servid="server_id"
g_form_machid="machine_id"
g_form_protover="protocol_version"
g_form_devid="device_id"
g_form_filedata="filedata"
g_form_defdata="defdata"
g_form_cfgdata="cfgdata"
g_form_prefdevid="preferred_devid"
g_form_reqfile="requested_file"
g_ctxt_zeroserv="127.0.0.1" # leave this one as localhost
g_ctxt_defserv=10.10.10.10
g_ctxt_csprefix="custom_script_"
g_ipar_expedite="expedite"
g_ipar_monitored="-monitored"
g_ipar_nomon="-nomon"
g_ipar_keeptalking="-keeptalking"
g_cursopt_nocheck="-k"
g_curlsopt_def=""
g_cmode_monitor="monitor"
g_cmode_upgrade="upgrade"
g_cmode_slave="slave"
g_cmode_single="single"
g_fn_bintarget="dwarfg_agent.sh"
g_fn_tgtdir="adwarfg_upgrade"
g_fn_upgtgt="adwarfg_upgrade.sh"
g_fn_uninst="uninstall.sh"
g_fn_cert="server.pem"
g_fn_newcert="server_new.pem"
g_fn_logger="logger"
g_fn_mysubdir="$$"
g_fn_tunkey="tunnel.key"
g_fn_logfile="logfile.txt"
g_fn_globlog="global_log.txt"
g_fn_webdata="web_export.src"
g_fn_defaults="defaults.tgz"
g_fn_cfgarch="cfgbackup.tgz"
g_fn_defmerge="defmerge.txt"
g_fn_newcfg="newconfig.txt"
g_fn_datafile="data.txt"
g_fn_tunnscript="webtunnel.sh"
g_fn_cfgimport="conf_import"
g_fn_settings="settings.ini" # regular settings file written to by agent
g_fn_srvsettings="server.ini" # generic server address file written to by e.g. web
g_fn_bcksettings="settings.ini.backup"
g_fn_resetbcksettings="settings.ini.backup.rst"
g_fn_cfgbackup="cfgbackup.txt"
g_fn_ocfgbackup="cfgbackup_old.txt"
g_fn_pidfile="${g_txtappname_short}.pid"
g_fn_tunnpidfile="${g_txtappname_short}_webtunnel.pid"
g_fn_childpidfile="${g_txtappname_short}_child.pid"
g_fn_devnull="/dev/null"
g_fn_cache="dwarfg-cache"
g_fn_counters="counters.txt"
g_fn_tmpdir="dwarfg-tmp"
g_floc_tmpdir="/tmp/$g_fn_tmpdir"
g_fn_cust_output="custom_script_output.txt"
g_fn_upgcopy="upg_tree"
g_fn_smallcycerr="adwarfg_smallerr.txt"
g_fn_bigcycerr="adwarfg_bigerr.txt"
g_fn_systemddir="/etc/systemd/system"
g_fn_sysvdir="/etc/init.d"
g_floc_appdir="/opt/adwarfg"
g_floc_certdir="/etc/ssl/certs"
g_floc_pidfile="$g_floc_appdir/$g_fn_pidfile"
g_floc_tunnpidfile="$g_floc_appdir/$g_fn_tunnpidfile"
g_floc_childpidfile="$g_floc_appdir/$g_fn_childpidfile"
g_floc_wterrfile="$g_floc_tmpdir/wterr.txt"
g_floc_settings="$g_floc_appdir/$g_fn_settings"
g_floc_setlock="$g_floc_settings.lck"
g_floc_counters="$g_floc_appdir/$g_fn_counters"
g_floc_counters_tgt="$g_floc_tmpdir/$g_fn_counters"
g_floc_webdata="$g_floc_appdir/$g_fn_webdata"
g_floc_webdata_tgt="$g_floc_tmpdir/$g_fn_webdata"
g_floc_upgrading="$g_floc_appdir/upgrading"
g_floc_monupgexit="$g_floc_appdir/upgexit"
g_floc_cbox="/usr/bin/cbox"
g_floc_csdir="$g_floc_appdir/custom_scripts"
g_floc_smallcycerr="$g_fn_devnull"
g_floc_bigcycerr="$g_fn_devnull"
g_floc_tunnscript="$g_floc_appdir/$g_fn_tunnscript"
g_floc_cfgimport="$g_floc_mydir/$g_fn_cfgimport"
g_floc_srvsettings="$g_floc_appdir/$g_fn_srvsettings"
g_floc_bcksettings="$g_floc_appdir/$g_fn_bcksettings"
g_floc_resetbcksettings="$g_floc_appdir/$g_fn_resetbcksettings"
g_floc_bintarget="$g_floc_appdir/$g_fn_bintarget"
g_floc_mydir="$g_floc_appdir/$g_fn_mysubdir"
g_floc_gdatadir="$g_floc_appdir/data"
g_floc_islppidf="/tmp/dwarfg_agent_isleep_pidfile"
g_fsuff_small=".small"
g_fsuff_big=".big"
g_fsuff_srv=".srv"
g_cmd_ramdrouter="mount -t ramfs ramdisk $g_floc_tmpdir"
g_cmd_ramdlinux="mount -t tmpfs -o size=16m tmpfs $g_floc_tmpdir"
g_curlsopt_useloccert="--cacert $g_floc_appdir/$g_fn_cert"
g_cts_datestart=$(date +%s)
g_fix_mindate="2024-05-10"
# WARNING! the following is hard-coded in server as well! Do not touch!
g_aerr_invid=1 # 0.1; Invalid ID
g_aerr_notex=2 # 0.1; Not-existent ID (on server)
g_aerr_regpr=3 # 0.1; Device registration (still) in progress
g_aerr_serfu=4 # 0.1; Server full OR license problem
g_aerr_serve=5 # 0.1; Server error (internal)
g_aerr_invda=6 # 0.1; Invalid data from client (agent)
g_aerr_datal=7 # 0.1; (Some) Data for device already waiting for processing on server (event locked)
g_aerr_datap=8 # 0.1; Data for device still being processed on server
g_aerr_comme=9 # 0.4; Communication error / protocol not understood (agent too new for old sever version)
g_aerr_neser=10 # 0.8; Not-Equal Server ID (agent contacted another server than the proper one)
# (same WARNING as above) data exchange files markup
g_cds_sectstart=":>>>>"
g_cds_sectend="<<<<:"
g_cds_lcsep=";"
g_cds_ssectstart=":--->"
g_cds_ssectend="<---:"
g_cds_sname_vars="VARS"
g_cds_sname_conf="CONF"
g_cds_sname_stat="STAT"
g_cds_sname_longstat="LONGSTAT"
g_cds_sname_cmds="CMDS"
g_cds_sname_cust="CUST"
g_cds_cmd_uptime="CMD_uptime"
g_cds_file_rconf="FILE_resolvconf"
g_cds_file_pppip="FILE_pppip"
g_cds_cmd_hostname="CMD_hostname"
g_cds_stat_sys="status sys"
g_cds_stat_lan="status lan"
g_cds_stat_net="netdev"
g_cds_stat_mwan="status mwan"
g_cds_stat_wifi="status wifi"
g_cds_stat_mobi="status mobile"
g_cds_stat_module="status module"
g_cds_stat_port="status ports"
g_cds_stat_vari="various"
g_cds_list_modules="user modules"
#g_cds_cmd_ip="IPaddr"
g_tuns_none=0
g_tuns_already=1
g_tuns_started=2
g_tuns_error=3
g_tuns_stopped=4
g_mindevidlen=4
g_maxdevidlen=4
g_devidlen=4
g_ctime_pingmin=4
g_time_certupderr=300
g_ctime_pingmax=600
g_ctime_pingdef=10
g_ctime_regmin=20
g_ctime_regmax=86400
g_ctime_regdef=260
g_ctime_regdbg=3
g_ctime_fallmin=20
g_ctime_fallmax=3600
g_ctime_falldef=60
g_ctime_falldbg=2
g_ctime_nextmin=0
g_ctime_nextmax=99999
g_ctime_nextdbg=1
g_ctime_err=600
g_ctime_err_shrt=240
g_ctime_atonce=2
g_cnum_smcycmin=1
g_cnum_smcycmax=100
g_cnum_smcycdef=5
g_cnum_conntolmin=0
g_cnum_conntolmax=100
g_cnum_conntoldef=5
g_cnum_regmax=100
# ERROR (message) levels. DO NOT TOUCH as this may *CRIPPLE* program completely (like start ignoring fatal errors out of sudden)
g_loglvl_dbg=1 # debug
g_loglvl_info=2 # info
g_loglvl_note=3 # log (notice)
g_loglvl_warn=4 # warning
g_loglvl_err=5 # error
g_loglvl_fatal=6 # fatal
g_loglvlmin=$g_loglvl_dbg # lowest log level for syn check from application
g_loglvlmax=$g_loglvl_fatal # highest log level for syn check from application
g_ctxt_initbody='
case "$1" in
	start)
		start
		;;
	stop)
		stop
		;;
	restart)
		stop
		start
		;;
	status)
		if [ -f "$g_floc_pidfile" ] ; then
			read -r input <$g_floc_pidfile
			if [[ -z "$input" || "$input" != "${input//[^0-9]}" ]] ; then
				echo "Rubbish in PIDfile ($g_floc_pidfile)" >&2
			elif [ "0" = "$input" ] ; then
				echo "Dwarfguard Agent is not running."
			elif [ -d "/proc/$input" ] ; then
				echo "Dwarfguard Agent is running (PID $input)."
			else
				echo "Dwarfguard Agent crashed/was killed (was PID $input)."
			fi
		else
			echo "Dwarfguard Agent PIDfile ($g_floc_pidfile) is missing." >&2
		fi
		;;
	*)
		echo "Dwarfguard Agent initscript. Usage: $g_fn_bintarget {start|stop|status|restart} (was: $0 $*)" >&2
		exit 1
		;;
esac
exit 0'
g_ctxt_copyr="copyright (c) Jan Otte, Dwarf Technologies s.r.o., 2023. Licensed under the BSD-like license - NO WARRANTY WHATSOEVER + modify (at your own risk) at will but keep this copyright line as first copyright + add line describing your modification and author identification"
g_ctxt_help="\n\
	$g_txtappname_long ... Agent for the Dwarfguard software (https://fossil.dwarftech.cz/mamaswiki/home).\n\n\
	Runs on the target device and provides information and other stuff to the server-side management software.\n\
	$g_ctxt_copyr\n\
	\n\
	Supported options:\n\
		-c          ... cleanup settings\n\
		-cache      ... store communication in $g_floc_tmpdir directory\n\
		                NOTE: only last data file sent is kept\n\
		                NOTE: only last non-empty data received (stripped) is kept\n\
		                WARN: the file kept may contain sensitive data!\n\
		-r          ... restore settings (after cleanup)\n\
		-d          ... debug (-keeptalking, -nomon, sets maximal verbosity for this session). \n\
		-D          ... DEBUG (like debug but also sets refresh interval to very fast, update $g_floc_settings after debug finished; NOTE: server may set the interval back). \n\
		-nomon      ... do not run in monitor mode\n\
		-keeptalking... do not redirect output (stdout+stderr) to /dev/null\n\
		-o          ... run one cycle and end (implies -nomon)\n\
		-omax <num> ... run <num> cycles and end (implies -nomon)\n"


# Variables - knobs for defaults
g_num_monsleep=10 # monitoring process sleep cycle (starting, increases +1 second every g_cnum_mfailinc agent exits
g_num_loopend=0 # 0 = perpetual, otherwise stops after the # of cycles
g_num_pingtries=3
g_cnum_mfailinc=5 # how many adwarfg failures before monitor sleep time is increased
g_forced_security= # if nonempty enforces this agent-hardcoded value irrelevant on settings
g_defsecurity=2 # set to 1 to allow certificate update (only when server known from past and matches g_cert_token (server IP change - YES, server reinstall/different server - NO), set to 0 to completely ignore certificate validity
g_security=${g_security:-$g_defsecurity}
g_custom_scripts_disabled=${g_custom_scripts_disabled:-}
[ -n "$g_forced_security" ] && g_security="$g_forced_security"
g_time_tunnerr="12" # tunnel sleep time after failure
g_sect_content=""
g_mode=$g_cmode_single
g_time_ping=$g_ctime_pingdef
g_time_reg=$g_ctime_regdef
g_time_fall=$g_ctime_falldef
g_time_tunnel=$g_time_fall
g_time_nextcycdef=$g_time_reg
g_num_smallcycles=$g_cnum_smcycdef
g_num_conntol=$g_cnum_conntoldef
g_alevel_debug=${g_alevel_debug:-$g_loglvl_note} # debug level value is $g_loglvl_note if not defined by environment (=LOG)
g_alevel_log=${g_alevel_log:-$g_loglvl_warn} # log level value is $g_loglvl_warn if not defined by environment (=WARNING)
g_alevel_syslog=${g_alevel_syslog:-$g_loglvl_fatal} # syslog level value is $g_loglvl_fatal if not defined by environment (=FATAL)

# Variables - other (not for changing)
g_childpid=0 # PID of real agent if we are performing monitoring role
g_ruser="root" # this cannot be changed after install (permissions set to user X will not change to user Y, the only possible change is to set from non-root user to root user (and never back)"
g_num_errors=0 # error counter
g_num_errors_printed=0 # last UI-communicated error counter
g_txt_lasterr="" # last error cache
#g_num_lastagent_uptime="" # last agent uptime - problem detection - initialize to empty string to detect no value read
#g_txt_lastfat="" # fatal cache
#g_txt_global_msg="" # Text to add to UI
g_fwfname=""
g_output=""
g_i_am_slave=""
g_url_register=""
g_url_dataexch=""
g_url_certupd=""
g_url_fwdown_conelos=""
g_url_fileget=""
g_settings_loaded=""
g_time_nextcyc=$g_time_nextcycdef
g_time_cycoverride=""
g_regfails=0
g_defs_sent=0
g_defs_sending=0
g_ecnt_small=0
g_ecnt_big=0
g_stat_small=
g_stat_big=
g_cert_token=${g_cert_token:-}
g_device_token=${g_device_token:-}
g_serverurl=${g_serverurl:-$g_ctxt_defserv}
g_floc_logfile=$g_floc_mydir/$g_fn_logfile
g_floc_glog=$g_floc_appdir/$g_fn_globlog
g_floc_cfgfile=$g_floc_gdatadir/$g_fn_cfgbackup
g_floc_oldcfgfile=$g_floc_gdatadir/$g_fn_ocfgbackup
g_floc_cfgarch=$g_floc_appdir/$g_fn_cfgarch
g_floc_tunkey=$g_floc_appdir/$g_fn_tunkey
g_snmppublic="public"
g_devid=""
g_prefdevid=""
g_servid=""
g_curlsopt_var="$g_curlsopt_def"
g_srvbase=""
g_fwdbase=""
g_fsuff_var="$g_fsuff_small"
g_cmd_ramd="$g_cmd_ramdlinux"
g_deployment=0
g_otherpid=0
g_tunstat=$g_tuns_none
g_txtbuff_incr=""
g_isleeppid=0
g_cycle=0
g_bigcyclenext=""
g_todo_restart=""
g_todo_tunnel=""
g_mysystem=""
g_shellmagic="#!/bin/bash"
g_initmagic=""
g_txt_monitor="" # nonempty when script should act as monitor process
# parameters
g_child_params=""
g_par_setdebug=""
g_par_cache=${g_par_cache:-} # set no cache (empty g_par_cache variable) if not defined by environment




# TODO:
# 1. improve work with data received from server:
#  -> data are structured in the same way as the data being send. Defensive parsing would be better.
#  -> process 'resend' flag from server - removes cache so that everything is resend (e.g. server lost data, requires all)
# 2. before sending data, compare every section with cache (if available). Do not send when identical. Repeat with subsections.
# 3. The lines from srv that were processed needs to be actually skipped, not parsed once more time.

isleep() { # interruptible (by signal) sleep; if 2 parameters are given, sleep PID is stored and target time exported
	local time_SEC time_MIN time_HOU time_DAY param time_now time_target
	param=$1
	if [ "$param" != "${param//[^0-9]}" ] ; then
		handle_me $g_loglvl_err "isleep argument NaN ($param)"
		return
	fi
	sleep "$1" &
	g_isleeppid=$!
	if [ -n "$2" ] ; then 
		echo "$g_isleeppid" >$g_floc_islppidf
		time_now=$(date "+%T")
		if [ "${time_now:6:1}" = "0" ] ; then
			time_SEC=$((${time_now:7:1}+param))
		else
			time_SEC=$((${time_now:6:2}+param))
		fi
		if [ "${time_now:3:1}" = "0" ] ; then
			time_MIN=$((${time_now:4:1}+time_SEC/60))
		else
			time_MIN=$((${time_now:3:2}+time_SEC/60))
		fi
		if [ "${time_now:0:1}" = "0" ] ; then
			time_HOU=$((${time_now:1:1}+time_MIN/60))
		else
			time_HOU=$((${time_now:0:2}+time_MIN/60))
		fi
		time_DAY=$((time_HOU/24))
		time_SEC=$((time_SEC%60))
		time_MIN=$((time_MIN%60))
		time_HOU=$((time_HOU%24))
		[ ${#time_SEC} -eq 1 ] && time_SEC="0$time_SEC";
		[ ${#time_MIN} -eq 1 ] && time_MIN="0$time_MIN";
		[ ${#time_HOU} -eq 1 ] && time_HOU="0$time_HOU";
		if [ "$time_DAY" -ne 0 ] ; then
			time_target="$time_HOU:$time_MIN:$time_SEC + $time_DAY days"
		else
			time_target="$time_HOU:$time_MIN:$time_SEC"
		fi
		echo "time_target=\"$time_target\"" >"$g_floc_webdata"
	fi
	wait $g_isleeppid # FIXME: on OpenWRT the wait seems to be uninterruptible (signals seems not being processed during wait in contrast to other shells behavior)
	[ -n "$2" ] && {
		cleanup_pending_synctime
		rm $g_floc_islppidf
	}
	g_isleeppid=0
}

msg() { # deliver a message (a debug/log/error message) where it should go; level prefix msg logfile
	local l_mylogfile
	l_mylogfile=${4:-$g_floc_logfile}
	if [ $# -lt 3 ] ; then
		echo "msg SYN ERR: called with invalid arguments ($#: $*)"
		return
	fi
	case $1 in
		''|*[!0-9]*)
			echo "msg SYN ERR: first parameter NaN ($1)"
			return
			;;
	esac
	if [[ "$l_mylogfile" != "$g_floc_logfile" && "$l_mylogfile" != "$g_floc_glog" ]] ; then # strange log file
		: # TODO: check, info to default log etc...
	fi
	if [ "$1" -ge "$g_alevel_debug" ]; then # anything at or above debug level is spit to stderr
		echo "$g_txtappname_short $2 $3" >&2
	fi
	[ "$1" -ge "$g_alevel_log" ] && {  # anything above or at log level is sent to logfile
		local l_datetime
		l_datetime=$(date "+%F %T")
		echo "$2" "[" "$l_datetime" "]" "$3" >>"$l_mylogfile"
	}
	if [ "$1" -ge "$g_alevel_syslog" ]; then # stuff above syslog level gets sent to syslog
		$g_fn_logger $g_txtappname_short "$2" "$3"
	fi
}

handle_me() { # handle Message or Error
	local l_globlog l_alevel l_ecounter
	l_ecounter=0
	if [[ $# -lt 2 || $# -gt 3 ]] ; then
		echo "handle_me SYN ERR: called with invalid arguments ($#)"
	fi
	if [ $# -eq 3 ] ; then
		l_globlog="yes please"
	else
		l_globlog=""
	fi
	case $1 in
		''|*[!0-9]*)
			echo "handle_me SYN ERR: first parameter NaN ($1)"
			return
			;;
	esac
	if [ "$1" -lt "$g_alevel_debug" ] ; then return; fi # ignore errors/msgs below active debuglevel
	case $1 in
		"$g_loglvl_dbg") # debug
			msg "$g_loglvl_dbg" "$g_txtll_debug" "$2" # DEBUG never goes to global logfile or syslog regardless settings
			;;
		"$g_loglvl_info") # info
			if [ "$g_alevel_syslog" -le "$g_loglvl_info" ] ; then l_alevel=$((g_alevel_syslog-1)); else l_alevel=$g_loglvl_info; fi # INFO never goes to syslog...
			msg "$l_alevel" "$g_txtll_info" "$2"
			;;
		"$g_loglvl_note") # log / note
			if [ "$g_alevel_syslog" -le "$g_loglvl_note" ] ; then l_alevel=$((g_alevel_syslog-1)); else l_alevel=$g_loglvl_note; fi # LOG /note never goes to syslog...
			msg "$l_alevel" "$g_txtll_note" "$2"
			[ -n "$l_globlog" ] && msg "$l_alevel" "$g_txtll_note" "$2" "$g_floc_glog" # global log posible from LOG/NOTE loglevel
			;;
		"$g_loglvl_warn") # warning
			msg $g_loglvl_warn $g_txtll_warn "$2"
			if [ -n "$l_globlog" ] ; then # global log...
				msg $g_loglvl_warn $g_txtll_warn "$2" $g_floc_glog
			fi
			;;
		"$g_loglvl_err") # error
			msg $g_loglvl_err $g_txtll_err "$2"
			g_num_errors=$((g_num_errors+1))
			g_txt_lasterr="$2"
			[ "$((g_num_errors_printed+g_err_print_tolerance))" -lt "$g_num_errors" ] && {
				dump_counters # communicate counter increase bigger than print tolerance to UI
				g_num_errors_printed="$g_num_errors"
			}
			if [ -n "$l_globlog" ] ; then # global log...
				msg $g_loglvl_err $g_txtll_err "$2" $g_floc_glog
			fi
			l_ecounter=$((l_ecounter+1))
			;;
		"$g_loglvl_fatal") # fatal
			if [ "$g_alevel_log" -gt "$g_loglvl_fatal" ] ; then l_alevel=$g_alevel_log; else l_alevel=$g_loglvl_fatal; fi # FATAL goes at least to logfile always
			msg $l_alevel $g_txtll_fatal "$2"
			msg $l_alevel $g_txtll_fatal "$2" $g_floc_glog # fatal is always logged to the global log file
			process_cleanup "handle_me:fatal" 1
			;;
	esac
}

process_cleanup() {
	local l_ecode l_lvl
	l_ecode="$2"
	l_lvl=$g_loglvl_warn
	[[ -n "$l_ecode" && "$l_ecode" = "${l_ecode//[^0-9]}" ]] || l_ecode=0 # no exit code provided, use 0
	[ 0 -eq $l_ecode ] || l_lvl=$g_loglvl_err
	stop_webtunnel # always kill webtunnel if exiting...
	if [ "$l_lvl" -lt "$g_loglvl_err" ] ; then
		handle_me $l_lvl "$g_txtappname_long version $g_appversion (mode: $g_mode) PID $$ is going down ($1) ..." "$g_floc_glog" # NOTE: prevent feedback loop with handle_me by testing loglvl first - global
	elif [ "$l_lvl" -eq "$g_loglvl_err" ] ; then
		handle_me $l_lvl "$g_txtappname_long version $g_appversion (mode: $g_mode) PID $$ is going down ($1) ..." "$g_floc_glog" # NOTE: prevent feedback loop with handle_me by testing loglvl first - global
	fi
	[[ 0 -eq $g_otherpid && -z "$g_i_am_slave" ]] && echo "0" >$g_floc_pidfile
	[[ 0 -ne "$g_isleeppid" && -d "/proc/$g_isleeppid" ]] && {
		kill "$g_isleeppid"
		[ -f "$g_floc_islppidf" ] && rm "$g_floc_islppidf"
	}
	[[ $g_childpid -ne 0 && -d "/proc/$g_childpid" ]] && {
		kill $g_childpid
		sleep $g_ctime_childexitwait
		[ -d "/proc/$g_childpid" ] && handle_me $g_loglvl_warn "Monitor: kill -9 child agent $g_childpid (still running after $g_ctime_childexitwait seconds)." && kill -9 $g_childpid # FIXME: this -9 kill is real on OpenWRT as other platform's default shell seems to be able to handle signals received during 'wait'
	}
	[ "$g_mode" = "$g_cmode_upgrade" ] && rm "$g_floc_upgrading" 2>$g_fn_devnull
	exit $l_ecode
}

initd_integration() {
	if [ -d "$g_fn_systemddir" ] ; then
		cat > $g_fn_systemddir/adwarfg.service <<EOF
[Unit]
Description=Agent for Dwarfguard system

[Service]
User=$g_ruser
WorkingDirectory=$g_floc_appdir
ExecStart=$g_floc_bintarget
Restart=always

[Install]
WantedBy=multi-user.target
EOF
		systemctl daemon-reload
		systemctl enable adwarfg.service || handle_me $g_loglvl_fatal "Failed to enable Dwarfguard Agent with systemd."
		return $g_deployd_systemd
	elif [ -d "$g_fn_sysvdir" ] ; then
		cat > $g_fn_sysvdir/adwarfg <<EOF
$g_shellmagic $g_initmagic
g_floc_pidfile="$g_floc_pidfile"
g_fn_bintarget="$g_fn_bintarget"
start() {
	$g_floc_bintarget &
}
stop() {
	if [ -f \$g_floc_pidfile ] ; then
		read -r input <\$g_floc_pidfile
		if [[ -z "\$input" || "\$input" != "\${input//[^0-9]}" ]] ; then
			echo "Unable to detect Dwarfguard agent PID (invalid number in \$g_floc_pidfile), no process was killed." >&2
		else
			if [ 0 -eq "\$input" ] ; then
				echo "Agent is not running." >&2
			else
				if [ -d "/proc/\$input" ] ; then
					local l_myvar l_match
					$g_ctxt_matchdetect
					if [ -n "\$l_match" ] ; then
						kill "\$input"
						sleep 4
						[ -d "/proc/\$input" ] && echo "Force-kill dwarfg, note that monitored children may exit at their next cycle" >&2 && kill -9 "\$input"
						echo "0" >"\$g_floc_pidfile"
						echo "Dwarfguard agent with PID \$input was killed."
					else
						echo "Agent process \$input is no longer running (another than Dwarfguard Agent process running with this PID)" >&2
					fi
				else
					echo "Agent process \$input is no longer running." >&2
				fi
			fi
		fi
	fi
}
$g_ctxt_initbody
EOF
		chmod a+x "$g_fn_sysvdir/adwarfg"
		if [[ "$g_mysystem" = "$g_sys_owrt" || "$g_mysystem" = "$g_sys_teltonika" ]] ; then
			"$g_fn_sysvdir/adwarfg" enable
		else
			for i in 2 3 4 5; do
				ln -s "$g_fn_sysvdir/adwarfg" "/etc/rc$i.d/S99adwarfg"
			done
		fi
		return $g_deployd_sysv
	else
		echo "Dwarfguard Agent script is NOT integrated within systemd or initd (none detected), one-time-run only. Provide your own boottime startup script executing just \"$g_floc_bintarget &\"" >&2
		return $g_deployd_own
	fi
}

deploy() {
	local l_rc l_upgrade
	l_rc=0
	l_upgrade=$1
	detect_system "drop" # system detection dropping the existin defined variable
	# the deployment itself (copy stuff into /opt/adwarfg)
	[ -z "$l_upgrade" ] && { # if not upgrading, create dir if not there, copy agent and perform linking
		mkdir -p $g_floc_appdir || handle_me $g_loglvl_fatal "Unable to create appdir ($g_floc_appdir)."
		cp "$0" $g_floc_appdir/$g_fn_bintarget || handle_me $g_loglvl_fatal "Unable to copy agent to appdir ($g_floc_appdir)."
		ln -s "$g_floc_counters_tgt" "$g_floc_counters"
		ln -s "$g_floc_webdata_tgt" "$g_floc_webdata"
	}
	[ -f "${0%/*}/$g_fn_cert" ] && { # copy/update cert in any mode
		cp "${0%/*}/$g_fn_cert" $g_floc_appdir/ || handle_me $g_loglvl_fatal "Unable to copy server certificate to appdir ($g_floc_appdir)" "$g_floc_glog"
	}
	[ -n "$l_upgrade" ] && { # upgrade-only code
		[[ -d "${0%/*}/$g_fn_upgcopy" && -n "$(ls -A "${0%/*}/$g_fn_upgcopy")" ]] && {
			cp -r "${0%/*}/$g_fn_upgcopy/"* "$g_floc_appdir/"
		}
		[ -f "$g_floc_counters" ] && {
			rm "$g_floc_counters"
				ln -s "$g_floc_counters_tgt" "$g_floc_counters"
		}
		[ -f "$g_floc_webdata" ] && {
			rm "$g_floc_webdata"
				ln -s "$g_floc_webdata_tgt" "$g_floc_webdata"
		}
	}
	case "$g_mysystem" in
		"$g_sys_icros"|"$g_sys_cstech")
			[ -z "$l_upgrade" ] && handle_me $g_loglvl_err "Deploy called on router system." "$g_floc_glog"
			return 0
			;;
		"$g_sys_owrt"|"$g_sys_teltonika")
			l_finame=$(basename "$0")
			[[ -z "$l_upgrade" && "$l_finame" != "postinst" && "$l_finame" != "adwarfg.postinst" ]] && { # do not run opkg when installing itself via opkg
				handle_me $g_loglvl_warn "Attempting to call opkg (should happen only with cmdline package), was called like this: \"$0\""
				opkg update
				opkg install curl || handle_me $g_loglvl_fatal "Unable to install curl, Dwarfguard Agent is not able to work without it." "$g_floc_glog"
			}
			initd_integration
			l_rc=$?
			;;
		"$g_sys_linux")
			initd_integration
			l_rc=$?
			;;
		*)
			handle_me $g_loglvl_fatal "Unable to detect system, Dwarfguard Agent will not work." "$g_floc_glog"
			;;
	esac
	cat >$g_floc_appdir/$g_fn_uninst <<EOF
$g_shellmagic
g_deployment="$l_rc"
g_floc_appdir="$g_floc_appdir"
case "\$g_deployment" in
	$g_deployd_systemd)
		echo "Stopping, disabling and removing Dwarfguard Agent from systemd..."
		systemctl stop adwarfg && systemctl disable adwarfg && systemctl daemon-reload
		rm "$g_fn_systemddir/adwarfg.service"
		;;
	$g_deployd_sysv)
		echo "Stopping and removing Dwarfguard Agent from initd..."
		"$g_fn_sysvdir/adwarfg" stop
		for i in 2 3 4 5 ""; do rm /etc/rc$i.d/S[0-9][0-9]adwarfg 2>/dev/null; done
		rm "$g_fn_sysvdir/adwarfg"
		;;
	*)
		echo "Unclear integration with init daemon, not running any integration cleanup."
		;;
esac
echo "Removing application from system..."
rm -rf "$g_floc_appdir"
EOF
	chmod u+x "$g_floc_appdir/$g_fn_uninst"
	# TODO non-root: solve the users first (if g_ruser is different than root, it must be created!)
	# TODO non-root: do not forget to chown appropriate dirs to g_ruser so that is can write to logs, create subdirs etc.
	[ -z "$l_upgrade" ] && return "$l_rc"
	return 0
}

expedite() {
	local l_maxc l_spid
	l_maxc=4
	while [ 0 -lt "$l_maxc" ] ; do
		if [ -f "$g_floc_islppidf" ] ; then
			IFS= read -r l_spid <"$g_floc_islppidf"
			[[ -n "$l_spid" && "$l_spid" = "${l_spid//[^0-9]}" && -d /proc/$l_spid ]] && ps | grep "$l_spid" | grep "sleep" >$g_fn_devnull && kill "$l_spid" && echo "Main waiting cycle has been terminated." && break
		fi
		echo "Interruptible sleep (main sleep only) not found, waiting ($l_maxc)..."
		sleep 1
		l_maxc=$((l_maxc-1))
	done
	return 0
}

badiff() {
	local l_diff l_lc l_readerrs l_linea l_lineb
	l_lc=0
	[ $# -ne 2 ] && return 255 # exactly two param needs to be provided
	[[ -f "$1" && -f "$2" ]] || return 254 # and it must be files
	while true
	do
		l_lc=$((l_lc+1))
		l_readerrs=0
		IFS= read -r l_linea <&3 || l_readerrs=1
		IFS= read -r l_lineb <&4 || l_readerrs=$((l_readerrs+1))
		[ $l_readerrs -eq 1 ] && l_diff="x" && break # one read err means one of the files ended prematurely
		[ $l_readerrs -eq 2 ] && break # two read errors - both files ended at the same time
		[ "$l_linea" != "$l_lineb" ] && l_diff="x" && break
	done 3<"$1" 4<"$2"
	[ -z "$l_diff" ] && return 0
	return 1
}

agent_upgrade() { # function performing upgrade to this (new) agent
	# this function could perform checks and other various stuff but in the end it boils down to...
	# TODO check rights when running under different user than root
	# TODO handle corner case like tunnel running
	local l_tmpsrc l_rcc l_mon_too_old
	l_tmpsrc="$g_floc_appdir/dwarfg_upg_src"
	[ -d "$g_floc_mydir" ] || mkdir -p "$g_floc_mydir"
	handle_me $g_loglvl_warn "Attempting to perform agent upgrade on agent start, I'm $0 version $g_appversion" "$g_floc_glog"
	cd "$g_floc_appdir" || {
		handle_me $g_loglvl_err "Error during upgrade - cannot cd to $g_floc_appdir" "$g_floc_glog"
		return 1
	}
	# capture devid, machid, servid, tokens and other important stuff from existing agent
	{
		grep "^g_devid=" <$g_floc_settings | head -1
		grep "^g_machid=" <$g_floc_settings | head -1
		grep "^g_deployment=" <$g_floc_settings | head -1
		grep "^g_servid=" <$g_floc_bintarget | head -1
		grep "^g_ctxt_defserv=" <$g_floc_bintarget | head -1
		grep "^g_appversion=" <$g_floc_bintarget | head -1
		grep "^g_[a-zA-Z]*_token=" <$g_floc_bintarget
	} >"$l_tmpsrc"
	sed -i "s/^g_appversion=/g_oldappversion=/" "$l_tmpsrc"
	. "$l_tmpsrc"
	case "$g_oldappversion" in
		"0.8.9"|"0.9.0"|"0.9.1"|"0.9.2"|"0.9.3")
			l_mon_too_old="yes"
			;;
		"0.9.4"|"0.9.5"|"0.9.6"|"0.9.7"|"0.9.8"|"0.9.9"|"1.0.0")
			:
			;;
		"$g_appversion")
			handle_me $g_loglvl_warn "Upgrading to the same version." "$g_floc_glog"
			;;
		*)
			handle_me $g_loglvl_fatal "Upgrade error - unsupported agent version $g_oldappversion." "$g_floc_glog"
			;;
	esac
	sed -i "s/^g_devid=.*/g_devid=\"$g_devid\"/" "$0"
	sed -i "s/^g_machid=.*/g_machid=\"$g_machid\"/" "$0"
	sed -i "s/^g_servid=.*/g_servid=\"$g_servid\"/" "$0"
	sed -i "s/^g_ctxt_defserv=.*/g_ctxt_defserv=\"$g_ctxt_defserv\"/" "$0"
	#cp -fr "$g_fn_tgtdir/*" "$g_floc_appdir/" || {
	#	handle_me $g_loglvl_err "Upgrade error - copying $g_fn_tgtdir to $g_floc_appdir"
	#	return 1
	#}
	deploy "upgrade" || handle_me $g_loglvl_fatal "Upgrade error - unable to update deployment." "$g_floc_glog"
	handle_me $g_loglvl_note "Agent deployment was updated." "$g_floc_glog"
	cp "$0" "$g_floc_bintarget" || {
		handle_me $g_loglvl_fatal "Upgrade error - renaming $0 to $g_floc_bintarget" "$g_floc_glog"
		return 1
	}
	# possibly: parse and update settings.ini and server.ini (if needed now, happens on first agent run)
	chmod a+x "$g_floc_bintarget" 2>$g_fn_devnull
	if [ "$g_deployment" = "$g_deployd_systemd" ] ; then
		# return immediately once upgrade done
		handle_me $g_loglvl_warn "Agent upgrade finished." 0
		return 0
	else
		# need to wait for the monitor to exit. NOTE: monitor should exit promptly
		handle_me $g_loglvl_note "Non-systemd deployment, upgrade waiting for monitor to exit..." "$g_floc_glog"
		sleep 4
		m_counter=0
		while [[ ! -e "$g_floc_monupgexit" && $m_counter -lt 6 ]] ; do
			m_counter=$((m_counter+1))
			sleep 2
		done
		[[ -e "$g_floc_monupgexit" || -n "$l_mon_too_old" ]] || handle_me $g_loglvl_err "Upgrade: monitor exit indication still not there after waiting cycle..." "$g_floc_glog"
		case "$g_deployment" in
			"$g_deployd_sysv")
				$g_fn_sysvdir/adwarfg start || handle_me $g_loglvl_fatal "Failed to start adwarfg via initd." "$g_floc_glog"
				;;
			"0")
				if [[ "$g_mysystem" = "$g_sys_icros" || "$g_mysystem" = "$g_sys_cstech" ]] ; then
					$g_floc_appdir/etc/init start || {
						handle_me $g_loglvl_err "Agent upgrade failure: unable to start new agent ($g_floc_bintarget)" "$g_floc_glog"
						rm "$l_tmpsrc" 2>$g_fn_devnull
						return 1
					}
				else
					handle_me $g_loglvl_fatal "Agent upgrade failure: the deployment ($g_deployment) is not set." "$g_floc_glog"
				fi
				;;
			"$g_deployd_own")
				if command -v nohup ; then
					nohup $g_floc_bintarget >$g_fn_devnull 2>&1 </dev/null &
					l_rcc=$?
				else
					$g_floc_bintarget >$g_fn_devnull 2>&1 <dev/null &
					l_rcc=$?
				fi
				[ $l_rcc -eq 0 ] || {
					handle_me $g_loglvl_err "Agent upgrade failure: unable to start new agent ($g_floc_bintarget)" "$g_floc_glog"
					rm "$l_tmpsrc" 2>$g_fn_devnull
					return 1
				}
				;;
			*)
				handle_me $g_loglvl_fatal "Agent upgrade failure: the deployment ($g_deployment) is not recognized." "$g_floc_glog"
				;;
		esac
	fi
	handle_me $g_loglvl_warn "Agent upgrade finished." 0
	return 0
}

add_tobuff() { # requires 4 parameters: start and end markers, name and content
	local l_smarker l_emarker l_secname l_seccont l_sechead l_seclen l_prefix
	if [ 4 -ne $# ]; then
		handle_me $g_loglvl_err "addsec_tobuff() called with incorrect number of parameters ($#)"
		return
	fi
	l_smarker="$1"
	l_emarker="$2"
	l_secname="$3"
	l_seccont="$4"
	if [ -z "$l_seccont" ] ; then
		return # no data in section, not going to add anything
	fi
	l_seclen=$(echo -e "$l_seccont" | grep -c "^")
	l_sechead="$l_smarker $l_secname$g_cds_lcsep$l_seclen $l_emarker"
	if [ -n "$g_txtbuff_incr" ] ; then
		l_prefix="$g_txtbuff_incr\n"
	else
		l_prefix=""
	fi
	g_txtbuff_incr="$l_prefix$l_sechead\n$l_seccont"
}

add_ssect_tobuff() {
	if [ 2 -ne $# ] ; then
		return
	fi
	add_tobuff "$g_cds_ssectstart" "$g_cds_ssectend" "$1" "$2"
}

add_sect_tobuff() {
	if [ 2 -ne $# ] ; then
		return
	fi
	add_tobuff "$g_cds_sectstart" "$g_cds_sectend" "$1" "$2"
}

run_custscripts() {
	local l_count l_failures l_scriptnames l_output l_output_nums l_output_txts l_output_flos
	l_failures=0
	l_count=$(ls $g_floc_csdir/$g_ctxt_csprefix* 2>/dev/null | wc -l)
	if [ 0 -lt "$l_count" ] ; then
		rm -f $g_floc_tmpdir/$g_fn_cust_output || return 1
		for i in $g_floc_csdir/$g_ctxt_csprefix*; do
			$i 2>"$g_floc_tmpdir/$(basename $i).errlog" >>"$g_floc_tmpdir/$g_fn_cust_output" || l_failures=$((l_failures+1))
			if [ -z "$l_scriptnames" ] ; then
				l_scriptnames="script=$i"
			else
				l_scriptnames="$l_scriptnames\nscript=$i"
			fi
		done
		l_output="$l_scriptnames"
		l_output_nums=$(grep "^custom_num[1-9]=[0-9]*$" <"$g_floc_tmpdir/$g_fn_cust_output")
		l_output_txts=$(grep "^custom_txt[1-9]=\".*\"$" <"$g_floc_tmpdir/$g_fn_cust_output")
		l_output_flos=$(grep "^custom_flo[1-9]=\"*[0-9][0-9]*\.[0-9][0-9]*\"*$" <"$g_floc_tmpdir/$g_fn_cust_output")
		for i in "$l_output_nums" "$l_output_txts" "$l_output_flos"; do
			[ -n "$i" ] && l_output="$l_output\n$i"
		done
	fi
	echo "cust_scripts=$l_count"
	echo "cust_failures=$l_failures"
	[ -n "$l_output" ] && echo -n "$l_output"
	return 0
}

stat_net_linux() {
	for pat in "/sys/class/net/e[^r]*" "/sys/class/net/w*" "/sys/class/net/usb*" "/sys/class/net/br*" "/sys/class/net/ppp*"; do
		if ls $pat >$g_fn_devnull 2>&1 ; then
			for i in $pat; do
				name=$(echo "$i"|sed "s/.*\///");
				for j in address statistics/collisions statistics/rx_bytes statistics/rx_packets statistics/rx_errors statistics/tx_bytes statistics/tx_packets statistics/tx_errors; do
					echo -n "$name/$j: "
					cat "$i"/"$j";
				done
				IP=$(ip -o -4 addr show dev "$name" | head -1 | sed "s#[^/]* \([0-9.]*/[0-9]*\).*#\1#")
				echo "$name/IPv4: $IP"
				IP=$(ip -o -6 addr show scope global dev "$name" | head -1 | sed "s#.*inet6 \([a-f0-9:]*/[0-9]*\).*#\1#")
				echo "$name/IPv6: $IP"
			done
		fi
	done
}

stat_various_linux() {
	local l_tmp l_tmptwo l_snmpname
	l_tmp=$(cat /proc/uptime)
	echo "uptime=\"$l_tmp\""
	l_tmp=$(grep MemTotal </proc/meminfo | sed "s/[^0-9]*\([0-9]*\).*/\1/")
	echo "memtotal=$l_tmp"
	l_tmp=$(grep MemAvailable </proc/meminfo | sed "s/[^0-9]*\([0-9]*\).*/\1/")
	echo "memavail=$l_tmp"
	case $g_mysystem in
		"$g_sys_icros"|"$g_sys_owrt"|"$g_sys_teltonika")
			l_tmp=$(ps | wc -l)
			l_tmptwo=$(df / | sed -e 1d | sed "s/[^0-9]*[0-9]*[^0-9]*[0-9]*[^0-9]*\([0-9]*\).*/\1/")
			;;
		"$g_sys_cstech")
			l_tmp=$(ps | wc -l)
			l_tmptwo=$(df /var/etc | sed -e 1d | sed "s/[^0-9]*[0-9]*[^0-9]*[0-9]*[^0-9]*\([0-9]*\).*/\1/")
			;;
		*)
			l_tmp=$(ps ax | wc -l)
			l_tmptwo=$(df --output=avail / | sed -e 1d)
			;;
	esac
	echo "runproc=$l_tmp"
	echo "rootavail=$l_tmptwo"
	l_tmp=$(cat /proc/loadavg)
	echo "loadavg=\"$l_tmp\""
	snmpget -V && {
		l_snmpname=$(snmpget -v 1 -c "$g_snmppublic" 127.0.0.1 .1.3.6.1.2.1.1.5.0 | grep "STRING:" | sed "s/^.*STRING:[ \t]*//")
		[ -n "$l_snmpname" ] && echo "snmpname=$l_snmpname"
	}
	echo "sysclock=$(date "+%s")"
}

stat_small_advrouter() { # gather small-cycle data (fast-gathered data)
	local l_str_comp l_str_tmp l_vars_sect l_stat_sect l_errtst l_pppiplen
	g_txtbuff_incr=""
	# prepare vars section first
	l_str_tmp=$(dump_settings 2>$g_floc_smallcycerr)
	l_str_comp="g_appversion=${g_appversion}\ng_fwfname=$g_fwfname\n${l_str_tmp}"
	add_sect_tobuff "$g_cds_sname_vars" "$l_str_comp"
	# store it
	l_vars_sect="$g_txtbuff_incr"
	# prep subsections in the stat section
	g_txtbuff_incr=""
	l_str_comp=$(status -v sys 2>>$g_floc_smallcycerr)
	add_ssect_tobuff "$g_cds_stat_sys" "$l_str_comp"
	l_str_comp=$(stat_net_linux 2>>$g_floc_smallcycerr)
	add_ssect_tobuff "$g_cds_stat_net" "$l_str_comp"
	l_str_comp=$(status -v lan 2>>$g_floc_smallcycerr)
	add_ssect_tobuff "$g_cds_stat_lan" "$l_str_comp"
	l_str_comp=$(status mwan 2>>$g_floc_smallcycerr)
	add_ssect_tobuff "$g_cds_stat_mwan" "$l_str_comp"
	l_str_comp=$(status mobile 2>$g_floc_bigcycerr)
	add_ssect_tobuff "$g_cds_stat_mobi" "$l_str_comp"
	l_str_comp=$(status wifi 2>>$g_floc_smallcycerr)
	add_ssect_tobuff "$g_cds_stat_wifi" "$l_str_comp"
	l_str_comp=$(cat /etc/resolv.conf 2>>$g_floc_smallcycerr)
	add_ssect_tobuff "$g_cds_file_rconf" "$l_str_comp"
	pppiplen=$(wc -c /var/ppp/ip 2>/dev/null| sed "s/\([0-9]*\).*/\1/")
	[[ -n "$pppiplen" && "0" != "$pppiplen" ]] && {
		l_str_comp=$(cat /var/ppp/ip 2>>$g_floc_smallcycerr)
		add_ssect_tobuff "$g_cds_file_pppip" "$l_str_comp"
	}
	l_str_comp=$(stat_various_linux 2>>$g_floc_smallcycerr)
	add_ssect_tobuff "$g_cds_stat_vari" "$l_str_comp"
	l_stat_sect="$g_txtbuff_incr"
	g_txtbuff_incr=""
	add_sect_tobuff "$g_cds_sname_stat" "$l_stat_sect"
	l_stat_sect="$g_txtbuff_incr"
	# handle cfg backup file
	backup >$g_floc_cfgfile 2>>$g_floc_smallcycerr
	if [ ! -f $g_floc_oldcfgfile ] || ! badiff "$g_floc_cfgfile" "$g_floc_oldcfgfile" ; then
		# no old file or old and new file is different
		handle_me $g_loglvl_info "Config change detected, preparing backup data..."
		cp "$g_floc_cfgfile" "$g_floc_oldcfgfile"
		tar czf "$g_floc_cfgarch" "$g_floc_cfgfile"
	fi
	l_errtst=$(grep -c "^" <"$g_floc_smallcycerr")
	if [ 0 -ne "$l_errtst" ] ; then
		g_ecnt_small=$((g_ecnt_small+1))
		handle_me $g_loglvl_note "Errors encountered during small run (# of lines: $l_errtst)."
	fi
	echo -en "$l_vars_sect\n$l_stat_sect" # check if this could expand the \n char right here
}

stat_small_linux() { # gather small-cycle data (fast-gathered data) (Linux for testing purposes only)
	local l_str_comp l_stat_sect l_str_set l_errtst
	g_txtbuff_incr=""
	# prep subsections in the stat section
	g_txtbuff_incr=""
	l_str_comp=$(uptime 2>>$g_floc_smallcycerr)
	add_ssect_tobuff "$g_cds_cmd_uptime" "$l_str_comp"
	l_str_comp=$(stat_net_linux 2>>$g_floc_smallcycerr)
	add_ssect_tobuff "$g_cds_stat_net" "$l_str_comp"
	l_str_comp=$(cat /etc/resolv.conf 2>>$g_floc_smallcycerr)
	add_ssect_tobuff "$g_cds_file_rconf" "$l_str_comp"
	if [ "$g_mysystem" = "$g_sys_teltonika" ] ; then
		l_str_comp="Temperature: $(gsmctl --temp 2>>$g_floc_smallcycerr)"
		add_ssect_tobuff "$g_cds_stat_sys" "$l_str_comp"
	fi
	l_str_comp=$(uname -n 2>$g_floc_smallcycerr)
	add_ssect_tobuff "$g_cds_cmd_hostname" "$l_str_comp"
	l_str_comp=$(stat_various_linux 2>>$g_floc_smallcycerr)
	add_ssect_tobuff "$g_cds_stat_vari" "$l_str_comp"
	l_stat_sect="$g_txtbuff_incr"
	g_txtbuff_incr=""
	# prepare variables
	l_str_set="g_appversion=$g_appversion\n$(dump_settings 2>$g_floc_smallcycerr)"
	# add variables section
	add_sect_tobuff "$g_cds_sname_vars" "$l_str_set"
	# add stat section
	add_sect_tobuff "$g_cds_sname_stat" "$l_stat_sect"
	l_errtst=$(grep -c "^" <"$g_floc_smallcycerr")
	if [ 0 -ne "$l_errtst" ] ; then
		g_ecnt_small=$((g_ecnt_small+1))
		handle_me $g_loglvl_note "Errors encountered during small run (# of lines: $l_errtst)."
	fi
	echo -n "$g_txtbuff_incr"
}

update_machid() { # common big-cycle stuff for all the agents
	local l_machid
	l_machid="$g_machid"
	get_machine_id
	[ "$l_machid" = "$g_machid" ] || save_settings "file"
}

stat_big_advrouter() { # gather big-cycle data (slow or resource-demanding gathering)
	local l_str_small l_str_comp l_str_stat l_errtst
	g_txtbuff_incr=""
	# prep subsections in the stat section
	g_txtbuff_incr=""
	l_str_comp=$(status module 2>>$g_floc_bigcycerr)
	add_ssect_tobuff "$g_cds_stat_module" "$l_str_comp"
	l_str_comp=$(status ports 2>>$g_floc_bigcycerr)
	add_ssect_tobuff "$g_cds_stat_port" "$l_str_comp"
	l_str_comp=$(for i in /opt/* ; do [[ -e "$i" && -f "$i/etc/version" ]] && { echo -n "${i##*/}: "; grep "" "$i/etc/version"; } ; done)
	[ -n "$l_str_comp" ] && add_ssect_tobuff "$g_cds_list_modules" "$l_str_comp"
	l_str_stat="$g_txtbuff_incr"
	# g_txtbuff_incr must be empty before calling small function as it uses the buff...
	g_txtbuff_incr=""
	l_str_small=$(stat_small_advrouter) # small cycle data included in big-cycle...
	# put it back to buff as we build on this one
	g_txtbuff_incr="$l_str_small"
	# add bigdata part - all in buff after this one
	add_sect_tobuff "$g_cds_sname_longstat" "$l_str_stat"
	l_errtst=$(grep -c "^" <"$g_floc_bigcycerr")
	if [ 0 -ne "$l_errtst" ] ; then
		g_ecnt_big=$((g_ecnt_big+1))
		handle_me $g_loglvl_note "Errors encountered during big run (# of lines: $l_errtst)."
	fi
	echo -n "$g_txtbuff_incr"
}

stat_big_linux() { # gather big-cycle data (slow or resource-demanding gathering) (Linux for testing only)
	local l_str_small l_str_comp l_str_stat l_errtst
	# prep subsections in the stat section
	g_txtbuff_incr=""
	if [ "$g_mysystem" = "$g_sys_teltonika" ] ; then
		l_str_comp="IMEI: $(gsmctl --imei 2>>$g_floc_bigcycerr)"
		l_str_comp="$l_str_comp\nICCID: $(gsmctl --iccid 2>>$g_floc_bigcycerr)"
		add_ssect_tobuff "$g_cds_stat_module" "$l_str_comp"
	fi
	l_str_stat="$g_txtbuff_incr"
	g_txtbuff_incr="" # need to empty it before calling the small func...
	l_str_small=$(stat_small_linux) # small-cycle data are included in big cycle...
	g_txtbuff_incr="$l_str_small" # put it back as a prefix
	# add bigdata part - all in buff after this one
	add_sect_tobuff "$g_cds_sname_longstat" "$l_str_stat"
	l_errtst=$(grep -c "^" <"$g_floc_bigcycerr")
	if [ 0 -ne "$l_errtst" ] ; then
		g_ecnt_big=$((g_ecnt_big+1))
		handle_me $g_loglvl_note "Errors encountered during big run (# of lines: $l_errtst)."
	fi
	echo -n "$g_txtbuff_incr"
}

dump_counters() {
	local idate
	idate="$(($(date +%s)-g_cts_datestart))"
	{
		echo " # Ignore this file, this is just agent counters to be sourced by UI cgi script and gets overwritten periodically"
		echo "pcnt_errors=$g_num_errors"
		echo "pcnt_connfails=$g_connfail"
		echo "g_num_lastagent_uptime=$idate"
	} >$g_floc_counters
}

dump_settings() {
	local idate
	idate="$(($(date +%s)-g_cts_datestart))"
	echo "g_alevel_debug=$g_alevel_debug"
	echo "g_alevel_log=$g_alevel_log"
	echo "g_alevel_syslog=$g_alevel_syslog"
	echo "g_mysystem=\"$g_mysystem\""
	echo "g_time_ping=$g_time_ping"
	echo "g_time_reg=$g_time_reg"
	echo "g_time_fall=$g_time_fall"
	echo "g_time_nextcyc=$g_time_nextcyc"
	echo "g_num_smallcycles=$g_num_smallcycles"
	echo "g_num_conntol=$g_num_conntol"
	echo "g_defs_sent=$g_defs_sent"
	echo "g_deployment=$g_deployment"
	echo "g_tunstat=$g_tunstat"
	echo "g_num_lastagent_uptime=$idate"
	echo "g_machid=\"$g_machid\""
	[ -n "$1" ] && {
		echo "g_devid=\"$g_devid\""
		echo "g_bigcyclenext=$g_bigcyclenext"
		echo "g_todo_tunnel=$g_todo_tunnel"
		echo "g_security=$g_security"
		echo "g_custom_scripts_disabled=\"$g_custom_scripts_disabled\""
		[ "$g_ctxt_fullvardump" = "$1" ] && { # dump tokens only on full dump
			echo "g_device_token=\"$g_device_token\""
			echo "g_cert_token=\"$g_cert_token\""
		}
	}
}

save_settings() {
	local i locked
	if [ $# -ne 0 ] ; then # really mess with the settings file
		# lock
		ln -s "$g_floc_settings" "$g_floc_setlock" || {
			i=0
			while [ $i -lt 5 ] ; do
				i=$((i+1))
				if ln -s "$g_floc_settings" "$g_floc_setlock" ; then
					locked="yep"
					break
				else
					sleep 1
				fi
			done
			[ -n "$locked" ] || {
				rm "$g_floc_setlock"
				handle_me $g_loglvl_warn "Removed defunct settings file lock" $g_floc_glog
				ln -s "$g_floc_settings" "$g_floc_setlock"
			}
		}
		case $1 in
			"RESET") # reset settings (cmdline request)
				handle_me $g_loglvl_note "CMDline: Settings reset" "$g_floc_glog"
				[ -f $g_floc_settings ] && mv $g_floc_settings $g_floc_bcksettings
				;;
			"RESTORE") # restore settings (cmdline request)
				handle_me $g_loglvl_note "CMDline: Settings restore" "$g_floc_glog"
				[ -f $g_floc_bcksettings ] && {
					[ -f $g_floc_settings ] && mv $g_floc_settings $g_floc_resetbcksettings
					mv $g_floc_bcksettings $g_floc_settings
				}
				;;
			*) # store current settings (overwrites)
				dump_settings "$g_ctxt_fullvardump" >$g_floc_settings
				if [ ! -f "$g_floc_srvsettings" ] && [ "$g_serverurl" != "$g_ctxt_zeroserv" ] ; then
					echo "g_serverurl=$g_serverurl" >"$g_floc_srvsettings"
				fi
				;;
		esac
		# unlock
		rm "$g_floc_setlock" 2>$g_fn_devnull
	else # do not mess with file, just dump settings out
		dump_settings "yes" # mid-full dump
	fi
}

load_settings() {
	if [ -f $g_floc_settings ] ; then
		. $g_floc_settings
	fi
	if [ -f $g_floc_srvsettings ] ; then # load server address if available
		. $g_floc_srvsettings
	fi
	[ -z "$g_settings_loaded" ] && g_settings_loaded="true"
	# now (re-)expand stuff dependent on loaded settings
	g_srvbase="https://$g_serverurl/dwarfg"
	g_fwdbase="https://$g_serverurl/firmware"
	g_url_register="$g_srvbase/d/r"
	g_url_dataexch="$g_srvbase/d/d"
	g_url_fileget="$g_srvbase/d/f"
	g_url_certupd="$g_srvbase/d/c"
	g_url_fwdown_conelos="$g_fwdbase/conelos"
}

machid_add_token() {
	local l_tok
	[[ $# -eq 1 && -n "$1" ]] || return 1
	l_tok="$1"
	l_res="${l_tok//[^0-9a-zA-Z_.]}"
	if [ -z "$m_machid_bit_wip" ] ; then
		m_machid_bit_wip="$l_res"
	else
		m_machid_bit_wip="${m_machid_bit_wip},$l_res"
	fi
}

machid_add_part() {
	[ $# -eq 1 ] && m_machid_bit_wip="${m_machid_bit_wip:0:$1}"
	if [ -z "$m_machid_wip" ] ; then
		m_machid_wip="$m_machid_bit_wip"
	else
		m_machid_wip="${m_machid_wip};$m_machid_bit_wip"
	fi
	m_machid_bit_wip=""
}

get_machine_id() {
	local l_tok
	m_machid_wip=""
	m_machid_bit_wip=""
	case $g_mysystem in
		"$g_sys_icros")
			l_tok="$(status sys -v | grep -i "^Serial Number" | sed "s/[^:]*:[ ]*//")"
			machid_add_token "$l_tok"
			machid_add_part 16 # serial
			l_tok="$(status lan -v | grep -i "^MAC Address" | sed "s/[^:]*:[ ]*//")"
			machid_add_token "$l_tok"
			machid_add_part 16 # MAC
			machid_add_part # disk (empty for Advantech router)
			l_tok="$(status sys -v | grep -i "^Product Name" | sed "s/[^:]*:[ ]*//")"
			machid_add_token "$l_tok"
			machid_add_part 32 # additional (product name)
			;;
		"$g_sys_linux")
			l_tok="$(if [ -r /var/lib/dbus/machine-id ] ; then cat /var/lib/dbus/machine-id; elif [ -r /var/db/dbus/machine-id ] ; then cat /var/db/dbus/machine-id; fi)"
			machid_add_token "$l_tok"
			machid_add_part 32 # serial
			for l_tok in $(cat /sys/class/net/*/address | sort -n | grep -v "^[0:]*$"|head -3); do
				machid_add_token "$l_tok"
			done
			machid_add_part 36 # MAC
			l_tok="$(disk="$(df | grep "/$" | sed "s/^[^ ]*\/\([^ ]*\).*/\1/")" && find /dev/disk/by-uuid -type l -exec sh -c "readlink {} | grep -o $disk >/dev/null && basename {}" \;)"
			machid_add_token "$l_tok"
			machid_add_part 36 # disk
			machid_add_part # additional (empty for linux)
			;;
		"$g_sys_cstech")
			l_tok="$(status sys -v | grep -i "^Serial Number" | sed "s/[^:]*:[ ]*//")"
			machid_add_token "$l_tok"
			machid_add_part 16 # serial
			l_tok="$(status lan -v | grep -i "^MAC Address" | sed "s/[^:]*:[ ]*//")"
			machid_add_token "$l_tok"
			machid_add_part 16 # MAC
			l_tok="$(status sys -v | grep -i "^Hardware UUID" | sed "s/[^:]*:[ ]*//")"
			machid_add_token "$l_tok"
			machid_add_part 36 # HW UUID
			l_tok="$(status sys -v | grep -i "^Product Type" | sed "s/[^:]*:[ ]*//")"
			machid_add_token "$l_tok"
			machid_add_part 24 # additional (product type)
			;;
		"$g_sys_owrt")
			l_tok="$(cat /proc/cpuinfo | grep -i "^Serial" | head -1 | sed "s/[^:]*:[ ]*//")"
			machid_add_token "$l_tok"
			l_tok="$(cat /proc/cpuinfo | grep -i "^Revision" |head -1 | sed "s/[^:]*:[ ]*//")"
			machid_add_token "$l_tok"
			l_tok="$(cat /proc/cpuinfo | grep -i "^Hardware" | head -1 | sed "s/[^:]*:[ ]*//")"
			machid_add_token "$l_tok"
			machid_add_part 36 # serial
			l_tok="$(cat /sys/class/net/eth0/address)"
			machid_add_token "$l_tok"
			machid_add_part 16 # MAC
			machid_add_part # disk (empty for OWRT)
			l_tok="$(cat /proc/sys/kernel/hostname)"
			machid_add_token "$l_tok"
			machid_add_part 24 # additional - hostname
			;;
		"$g_sys_teltonika")
			l_tok="$(grep "option serial" /etc/config/hwinfo | sed "s/.*'\(.*\)'.*/\1/")"
			machid_add_token "$l_tok"
			l_tok="$(grep "option mnf_code" /etc/config/hwinfo | sed "s/.*'\(.*\)'.*/\1/")"
			machid_add_token "$l_tok"
			machid_add_part 32 # serial
			l_tok="$(cat /sys/class/net/eth0/address)"
			machid_add_token "$l_tok"
			machid_add_part 16 # MAC
			machid_add_part # disk (empty for Teltonika router)
			l_tok="$(cat /proc/sys/kernel/hostname)"
			machid_add_token "$l_tok"
			machid_add_part 24 # additional - hostname
			;;
		*)
			;;
	esac
	g_machid="$m_machid_wip"
	return 0
}

device_register() {
	local l_pdev l_rest l_serv_response l_separators l_devid_new l_di_tmp l_delay l_errwait
	[ -z "$g_machid" ] && {
		get_machine_id
		[ -z "$g_machid" ] && {
			sleep 30
			handle_me $g_loglvl_fatal "Unable to generate machine ID."
		}
	}
	handle_me $g_loglvl_info "Registering device..."
	if [ -n "$g_prefdevid" ] ; then
		if [ -n "$g_devid" ] ; then
			handle_me $g_loglvl_info "Trying to use last known devid ($g_devid) as preferred devid for re-registration..."
			l_pdev="-F $g_form_prefdevid=$g_devid"
		else
			handle_me $g_loglvl_info "Trying to use pre-set devid ($g_prefdevid) as preferred devid for registration..."
			l_pdev="-F $g_form_prefdevid=$g_prefdevid"
		fi
	else
		l_pdev=""
	fi
	handle_me $g_loglvl_dbg "Register command is curl $g_curlsopt_var -s -S -F \"$g_form_servid=$g_servid\" -F \"$g_form_devtype=$g_mysystem\" -F \"$g_form_protover=$g_protoversion\" -F \"$g_form_machid=\\\"$g_machid\\\"\" $l_pdev \"$g_url_register\""
	l_serv_response=$(curl $g_curlsopt_var -s -S -F "$g_form_servid=$g_servid" -F "$g_form_devtype=$g_mysystem" -F "$g_form_protover=$g_protoversion" -F "$g_form_machid=\"$g_machid\"" $l_pdev "$g_url_register")
	process_curl_response "$?" || return 1
	handle_me $g_loglvl_dbg "Server response is \"$l_serv_response\""
	l_separators="${l_serv_response//[^$g_ctxt_devidsep]}"
	if [ 3 -gt ${#l_separators} ] ; then
		handle_me $g_loglvl_err "Unable to correctly parse server registration response."
		return 1
	fi
	l_devid_new="${l_serv_response%%${g_ctxt_devidsep}*}"
	l_rest="${l_serv_response#*${g_ctxt_devidsep}}"
	g_device_token="${l_rest%%${g_ctxt_devidsep}*}"
	l_rest="${l_rest#*${g_ctxt_devidsep}}"
	g_cert_token="${l_rest%%${g_ctxt_devidsep}*}"
	l_rest="${l_rest#*${g_ctxt_devidsep}}"
	if [ 3 -eq ${#l_separators} ] ; then
		l_delay="$l_rest"
	else
		l_delay="${l_rest%%${g_ctxt_devidsep}*}"
		handle_me $g_loglvl_note "Part of the server response was not parsed. Consider upgrading agent."
	fi
	[ "$l_param" != "${l_param//[^0-9]}" ] && l_delay=""
	l_di_tmp=$(echo -ne "$l_devid_new" |grep -c "^")
	if [ "$l_di_tmp" -ne 1 ] ; then
		handle_me $g_loglvl_err "Unable to register at the moment, posponed for another cycle..."
		handle_me $g_loglvl_dbg "Resp. was: $l_devid_new"
		isleep "$g_ctime_err_shrt"
		return 1
	fi
	case "$l_devid_new" in
		"$g_regerr_err")
			handle_me $g_loglvl_err "Registering device error - server full? Check application log on server."
			[ -z "$l_delay" ] && l_delay=$g_ctime_err
			isleep "$l_errwait"
			return 1
			;;
		"$g_regerr_machid")
			handle_me $g_loglvl_err "Machine ID is missing."
			[ -z "$l_delay" ] && l_delay=$g_ctime_err
			isleep "$l_delay"
			return 1
			;;
		"$g_regerr_license")
			handle_me $g_loglvl_err "Invalid license at server! Forcing sleep."
			[ -z "$l_delay" ] && l_delay=$g_ctime_err
			isleep "$l_delay"
			return 1
			;;
		"$g_regerr_noop")
			handle_me $g_loglvl_err "Registering device error - app reports it is in NO-OP state. Check server logs."
			[ -z "$l_delay" ] && l_delay=$g_ctime_err
			isleep "$l_delay"
			return 1
			;;
		"$g_regerr_comm")
			handle_me $g_loglvl_err "Registering device error - app reports unknown communication error (network error/server version obsolete?)."
			[ -z "$l_delay" ] && l_delay=$g_ctime_err_shrt
			isleep "$l_delay"
			return 1
			;;
		"$g_regerr_servid")
			handle_me $g_loglvl_err "Registering device error - server reports invalid Server ID (mismatched server address?)."
			[ -z "$l_delay" ] && l_delay=$g_ctime_err
			isleep "$l_delay"
			return 1
			;;
		"$g_regerr_devt")
			handle_me $g_loglvl_err "Registering device error - app reports device type ($g_mysystem) missing or invalid."
			[ -z "$l_delay" ] && l_delay=$g_ctime_err
			isleep "$l_delay"
			return 1
			;;
		*)
			[ -z "$g_device_token" ] && {
				handle_me $g_loglvl_err "Device agent token is empty, unable to accept device id"
				return 1
			}
			[ -z "$g_cert_token" ] && handle_me $g_loglvl_warn "Certificate token is empty, certificate exchange will not be possible."
		if echo "$l_devid_new" | grep "^0*$" ; then
			handle_me $g_loglvl_err "Server is returning void ID, check if not full. Registration failed."
			return 1
		fi
		handle_me $g_loglvl_dbg "New potential DEVICE_ID received: \"$l_devid_new\""
		l_di_tmp=${#l_devid_new}
		if [ "$l_di_tmp" -ne "$g_devidlen" ] ; then
			if [ "$l_di_tmp" -lt "$g_mindevidlen" ] ; then
				handle_me $g_loglvl_err "Device ID length incorrect AND extremly short, registration failed."
				return 1
			elif [ "$l_di_tmp" -gt "$g_maxdevidlen" ] ; then
				handle_me $g_loglvl_err "Device ID length incorrect AND extremly long, registration failed."
				return 1
			else
				handle_me $g_loglvl_fatal "Device ID length incorrect but within tolerance. Consider updating agent."
			fi
		fi
		if [ -z "$g_devid" ] ; then
			handle_me $g_loglvl_warn "Registered as $l_devid_new" "$g_floc_glog" # g_devid goes to log here
		else
			handle_me $g_loglvl_warn "Re-registered as $l_devid_new" "$g_floc_glog" # same as above
		fi
		g_devid="$l_devid_new"
		g_defs_sent=0 # new registration -> defaults need to be re-send
		rm -f "$g_floc_cfgfile" "$g_floc_oldcfgfile" 2>$g_fn_devnull # new registration -> backup config must be re-send
		save_settings "file" # remember updated settings
		g_cycle=$((g_num_smallcycles+1)) # run bigcycle next
		g_time_cycoverride=$g_ctime_atonce # run it immediately when available
		# if delay time requested, sleep NOW
		[ -n "$l_delay" ] && handle_me $g_loglvl_info "Server-requested sleep $l_delay..." && isleep "$l_delay"
		return 0
			;;
	esac
	handle_me $g_loglvl_err "Unprocessed registration error"
	isleep "$g_ctime_err"
	return 1
}

process_curl_response() {
	local l_param
	[ $# -lt 1 ] && return 1 # no curl return code provided
	l_param="$1"
	[ "$l_param" != "${l_param//[^0-9]}" ] && return 1 # parameter is NaN
	case "$1" in
		0)
			return 0
			;;
		58|59|60|77) # SSL and certificate problems
			handle_me $g_loglvl_err "Curl SSL error ($1)"
			if [ -n "$2" ] ; then
				cert_update
			fi
			return 1 # NOTE: return error as orig communication failed due to $1 being not 0 (even if the cert_update could success)
			;;
		27) # out-of-memory
			handle_me $g_loglvl_err "Out-of-memory error."
			isleep "$g_ctime_err"
			;;
		7) # connection error
			handle_me $g_loglvl_err "Unable to connect to server"
			isleep "$g_ctime_err_shrt"
			;;
		55|56) # network error
			handle_me $g_loglvl_err "Network error"
			isleep "$g_ctime_err_shrt"
			;;
		*) # other error
			handle_me $g_loglvl_err "Comm error: $1"
			isleep "$g_ctime_err_shrt"
			;;
	esac
	return 1
}

data_exchange() {
	# uploads data to server and get data back
	local l_estr_def l_estr_bck l_server_response l_lc_head l_lc_tail l_cmdrc l_lc_sr l_lc_ar l_lineskip l_linecount l_line l_line_target l_resp_code l_tmpline l_delay l_retcode l_app_resp l_sect_name l_sect_len l_sect_begline l_sect_endline
	handle_me $g_loglvl_dbg "data_exchange()"
	if [ -z "$g_output" ] ; then
		handle_me $g_loglvl_warn "Not sending empty data to application."
		return $g_errc_inval
	fi
	if [ -n "$g_par_cache" ] ; then
		handle_me $g_loglvl_dbg "Storing output in $g_floc_tmpdir/$g_fn_cache$g_fsuff_var"
		echo -ne "$g_output" >$g_floc_tmpdir/$g_fn_cache$g_fsuff_var
		[ 1 -eq $g_defs_sending ] && cp $g_floc_gdatadir/$g_fn_defaults $g_floc_tmpdir/
		[ -f "$g_floc_cfgarch" ] && cp "$g_floc_cfgarch" $g_floc_tmpdir/
	fi
	[[ 1 -eq $g_defs_sending && "$g_sys_owrt" != "$g_mysystem" && "$g_sys_teltonika" != "$g_mysystem" && "$g_sys_cstech" != "$g_mysystem" ]] && l_estr_def="-F $g_form_defdata=@$g_floc_gdatadir/$g_fn_defaults;filename=$g_fn_defaults" && handle_me $g_loglvl_dbg "Sending defaults" # TODO OpenWRT, Teltonika - unblock once defaults location are known
	[ -f "$g_floc_cfgarch" ] && l_estr_bck="-F $g_form_cfgdata=@$g_floc_cfgarch;filename=$g_fn_cfgarch" && handle_me $g_loglvl_dbg "Sending config backup"
	handle_me $g_loglvl_dbg "Dataexch command is echo -ne \"$g_output\" | curl -s -S -X POST $g_curlsopt_var -H \"Content-Type: multipart/form-data\" -w \"\n%{http_code}\" -F "$g_form_protover=$g_protoversion" -F \"$g_form_devid=$g_devid\" -F \"$g_form_machid=$g_machid\" -F \"$g_form_devtoken=$g_device_token\" -F \"$g_form_filedata=@-;filename=$g_fn_datafile\" $l_estr_def $l_estr_bck \"$g_url_dataexch\""
	l_server_response=$(echo -ne "$g_output" | curl -s -S -X POST $g_curlsopt_var -H "Content-Type: multipart/form-data" -w "\n%{http_code}" -F "$g_form_protover=$g_protoversion" -F "$g_form_devid=$g_devid" -F "$g_form_machid=\"$g_machid\"" -F "$g_form_devtoken=$g_device_token" -F "$g_form_filedata=@-;filename=$g_fn_datafile" $l_estr_def $l_estr_bck "$g_url_dataexch")
	l_cmdrc=$?
	handle_me $g_loglvl_dbg "Curl response is $l_cmdrc"
	[ 0 -eq "$l_cmdrc" ] || {
		handle_me $g_loglvl_err "Error running data exchange ($l_cmdrc)"
		return $g_errc_srv
	}
	process_curl_response "$l_cmdrc" "certupdate" || return $g_errc_comm
	l_lc_sr=$(echo -ne "$l_server_response"|grep -c "^")
	handle_me $g_loglvl_dbg "Got $l_lc_sr lines from server."
	handle_me $g_loglvl_dbg "Server response is: $l_server_response"
	l_resp_code=$(echo -ne "$l_server_response"|tail -1)
	handle_me $g_loglvl_dbg "HTTP response code is \"$l_resp_code\""
	case $l_resp_code in
		''|*[!0-9]*)
			handle_me $g_loglvl_err "Empty or NaN response code from server (\"$l_resp_code\")"
			return $g_errc_srvinval
			;;
		200)
			: # OK
			;;
		*)
			handle_me $g_loglvl_err "Non-ok HTTP response code received, discarding input. Resp. code: $l_resp_code"
			return $g_errc_srv
			;;
	esac
	l_tmpline=$(echo -ne "$l_server_response"|head -1)
	if [ "$l_tmpline" = "${l_tmpline//${g_ctxt_devidsep}}" ] ; then # only retcode - no devid_sep - no delay time
		l_retcode="$l_tmpline"
		l_delay=""
	else # response contains devid_set, parse stuff...
		l_retcode=${l_tmpline%%${g_ctxt_devidsep}*} # strip end (longest) (code:whatever:sleep time)
		l_delay=${l_tmpline##*${g_ctxt_devidsep}} # strip begin (longest) (code:whatever:sleep time)
	fi
	# if delay time requested and it is a number, sleep NOW
	[[ -n "$l_delay" && "$l_delay" = "${l_delay//[^0-9]}" ]] && handle_me $g_loglvl_info "Server-requested sleep $l_delay..." && isleep "$l_delay"
	handle_me $g_loglvl_dbg "Application return code is \"$l_retcode\""
	case $l_retcode in
		''|*[!0-9]*)
			handle_me $g_loglvl_err "Empty or NaN return code from application (\"$l_retcode\")"
			return $g_errc_appinval
			;;
		"$g_aerr_invid"|"$g_aerr_notex")
			handle_me $g_loglvl_warn "Application requested to re-register this device (Err:$l_retcode). Re-registering..."
			if device_register ; then
				return $g_errc_rereg # registered for now, data will be send on next cycle...
			else
				handle_me $g_loglvl_warn "Device registration failed, looping..."
				g_time_cycoverride=$g_time_fall
				return $g_errc_app
			fi
			;;
		"$g_aerr_regpr")
			g_regfails=$((g_regfails+1))
			if [ $g_regfails -eq $g_cnum_regmax ] ; then
				handle_me $g_loglvl_err "Maximum number of register-wait cycles reached. Re-registering..." "$g_floc_glog"
				g_regfails=0
				if device_register ; then
					return $g_errc_rereg # registered for now, data will be send on next cycle...
				else
					#handle_me $g_loglvl_err "Device registration failed, looping..."
					g_time_cycoverride=$g_time_fall
					return $g_errc_app
				fi
			else
				handle_me $g_loglvl_warn "Application reports this device registration is still in progress. Counter: $g_regfails, max: $g_cnum_regmax."
				return $g_errc_app
			fi
			;;
		"$g_aerr_serfu")
			handle_me $g_loglvl_err "Dwarfguard server full / licensing issue ($g_aerr_serfu). Please investigate."
			return $g_errc_app
			;;
		"$g_aerr_serve")
			handle_me $g_loglvl_warn "Application reports internal error ($g_aerr_serve). Please investigate."
			# there is no need in shorting down the time on app error
			return $g_errc_app
			;;
		"$g_aerr_invda")
			handle_me $g_loglvl_err "Application reports invalid data sent ($g_aerr_invda). Check if update needed."
			return $g_errc_app
			;;
		"$g_aerr_comme")
			handle_me $g_loglvl_err "Application reports communication error ($g_aerr_comme). Check if update needed."
			return $g_errc_app
			;;
		"$g_aerr_datal"|"$g_aerr_datap")
			handle_me $g_loglvl_err "Application reports data for this device already waits for processing ($l_retcode). Check server."
			return $g_errc_app
			;;
		"$g_aerr_neser")
			handle_me $g_loglvl_err "Improper server contacted / server uses different Server ID than stored in this agent. Fix server URL or update agent."
			return $g_errc_app
			;;
		0)
			: # OK
			;;
		*)
			handle_me $g_loglvl_err "Not understood application return code, discarding input. Possible manual agent upgrade is needed. Ret.code: $l_retcode"
			return $g_errc_app
			;;
	esac
	l_lc_head=$((l_lc_sr-1)) # skip app resp. code
	l_lc_tail=$((l_lc_head-1)) # skip HTTP resp. code
	if [ "$l_lc_tail" -gt 0 ] ; then # if application response is non-empty, process it
		l_app_resp=$(echo -n "$l_server_response" | head -$l_lc_head | tail -$l_lc_tail)
		l_linecount=0
		l_line_target=0
		l_lineskip=0
		l_lc_ar=$(echo -n "$l_app_resp" | grep -c "^")
		if [[ $l_lc_ar -gt 0 && -n "$g_par_cache" ]] ; then
			echo -n "$l_app_resp" >$g_floc_tmpdir/$g_fn_cache$g_fsuff_srv
		fi
		while IFS= read -r l_line
		do
			l_linecount=$((l_linecount+1))
			if [ $l_line_target -gt $l_linecount ] ; then # should we need to skip the already processed section...
				continue
			fi
			handle_me $g_loglvl_dbg "Processing line no. $l_linecount, content: \"$l_line\""
			case $l_line in
				'')
					continue # skip empty lines
					;;
				${g_cds_sectstart}[\ ]*${g_cds_lcsep}*[\ ]${g_cds_sectend})
					# new section start
					l_sect_name=${l_line#${g_cds_sectstart} } # strip begin (shortest)
					l_sect_name=${l_sect_name%%${g_cds_lcsep}*} # strip end (longest)
					l_sect_len=${l_line##*${g_cds_lcsep}} # strip begin (longest)
					l_sect_len=${l_sect_len% ${g_cds_sectend}} # strip end (shortest)
					handle_me $g_loglvl_info "Section \"$l_sect_name\" length: \"$l_sect_len\" received from application"
					if [ "$l_sect_len" != "${l_sect_len//[^0-9]}" ] ; then
						handle_me $g_loglvl_warn "Section \"$l_sect_name\" length is NaN. ignoring..."
						continue
					fi
					if [ 0 -eq "$l_sect_len" ] ; then
						handle_me $g_loglvl_warn "Invalid (zero-length) section obtained from application, ignoring..."
						continue
					fi
					l_sect_begline=$((l_linecount+1))
					l_sect_endline=$((l_linecount+l_sect_len))
					l_line_target=$((l_sect_endline+1)) # mark that we need to skip the section lines
					g_sect_content=$(echo -n "$l_app_resp" | sed -n "${l_sect_begline},${l_sect_endline}p;${l_line_target}q")
					handle_me $g_loglvl_info "Section content is: $g_sect_content"
					# match sect name
					case $l_sect_name in
						"$g_cds_sname_vars")
							update_vars
							;;
						"$g_cds_sname_conf")
							process_config
							;;
						"$g_cds_sname_cmds")
							process_commands
							;;
						"$g_cds_sname_cust")
							[ -z "$g_custom_scripts_disabled" ] && process_custom_scripts
							;;
						*)
							handle_me $g_loglvl_warn "Unknown section name received from application (\"$l_sect_name\"), skipping section..."
							continue
							;;
					esac
					#TODO/FIXME: *actually* skip the processed lines, no need to parse twice - NOTE the continue statements above, almost all have to skip section
					;;
				*)
					handle_me $g_loglvl_info "Skipping line from application: $l_line"
					l_lineskip=$((l_lineskip+1))
					;;
			esac
		done <<EOF
$l_app_resp
EOF
		if [ 0 -lt "$l_lineskip" ] ; then
			handle_me $g_loglvl_warn "Some lines from application were skipped. Number of lines: $l_lineskip"
		fi
	fi # end processing application response
	return 0
}

# check if the provided string is a number
# if min and max are provided, the number is checked against these bounds
number_ok() {
	local l_testnum
	[ $# -ge 1 ] || return 1
	l_testnum="$1"
	[ "$l_testnum" = "${l_testnum//[^0-9]}" ] || return 1
	[[ -n "$2" && "$l_testnum" -lt "$2" ]] && return 1
	[[ -n "$3" && "$l_testnum" -gt "$3" ]] && return 1
	return 0
}

update_vars() {
	local l_linecount l_touched l_touched_old l_newvar
	l_touched=0
	l_touched_old=0
	l_linecount=$(echo -ne "$g_sect_content" | grep -c "^")
	if [ 1 -gt "$l_linecount" ] ; then
		return 1
	fi
	handle_me $g_loglvl_info "Got $l_linecount new variables from application."
	while IFS= read -r l_newvar
	do
		handle_me $g_loglvl_dbg "Processing newvar line $l_newvar"
		local l_var_name l_var_value
		l_var_name=${l_newvar%%=*} # cut at first equal (longest str. from end)
		l_var_value=${l_newvar#*=} # cut at first equal (shortest str. from start)
		if [[ 1 -gt ${#l_var_name} || "$l_newvar" != "${l_newvar#[0-9]}" ]] ; then
			handle_me $g_loglvl_warn "Variable name empty or begins with number ($l_newvar)"
		else
			case $l_var_name in
				"g_time_ping")
					if number_ok "$l_var_value" "$g_ctime_pingmin" "$g_ctime_pingmax" ; then
						g_time_ping=$l_var_value
						l_touched=$((l_touched+1))
					fi
					;;
				"g_time_reg")
					if number_ok "$l_var_value" "$g_ctime_regmin" "$g_ctime_regmax" ; then
						g_time_reg=$l_var_value
						l_touched=$((l_touched+1))
					fi
					;;
				"g_time_fall")
					if number_ok "$l_var_value" "$g_ctime_fallmin" "$g_ctime_fallmax"; then
						g_time_fall=$l_var_value
						l_touched=$((l_touched+1))
					fi
					;;
				"g_time_nextcyc")
					if number_ok "$l_var_value" "$g_ctime_nextmin" "$g_ctime_nextmax" ; then
						g_time_nextcyc=$l_var_value
						g_time_cycoverride=$l_var_value
						l_touched=$((l_touched+1))
					fi
					;;
				"g_num_smallcycles")
					if number_ok "$l_var_value" "$g_cnum_smcycmin" "$g_cnum_smcycmax" ; then
						g_num_smallcycles=$l_var_value
						l_touched=$((l_touched+1))
					fi
					;;
				"g_num_conntol")
					if number_ok "$l_var_value" "$g_cnum_conntolmin" "$g_cnum_conntolmax" ; then
						g_num_conntol=$l_var_value
						l_touched=$((l_touched+1))
					fi
					;;
				"g_alevel_debug")
					if number_ok "$l_var_value" "$g_loglvlmin" "$g_loglvlmax"; then
						g_alevel_debug=$l_var_value
						l_touched=$((l_touched+1))
					fi
					;;
				"g_alevel_log")
					if number_ok "$l_var_value" "$g_loglvlmin" "$g_loglvlmax"; then
						g_alevel_log=$l_var_value
						l_touched=$((l_touched+1))
					fi
					;;
				"g_alevel_syslog")
					if number_ok "$l_var_value" "$g_loglvlmin" "$g_loglvlmax"; then
						g_alevel_syslog=$l_var_value
						l_touched=$((l_touched+1))
					fi
					;;
				*)
					handle_me $g_loglvl_warn "Ignoring unsupported variable set from app: $l_var_name"
					;;
					# NOTE support for enforcing sending defaults again could come here
			esac
			if [ "$l_touched" -eq "$l_touched_old" ] ; then
				handle_me $g_loglvl_warn "Var-setting line from application failed: $l_newvar"
			else
				l_touched_old=$l_touched
				handle_me $g_loglvl_info "New VAR value from application: $l_newvar"
			fi
		fi
	done <<EOF
$(echo -e "$g_sect_content")
EOF
	if [ 0 -ne $l_touched ] ; then
		handle_me $g_loglvl_note "New values from application processed ($l_touched)"
		save_settings "file"
	fi
}

process_config() {
	# TODO: 1) syntax check for config content
	# TODO: 2) store existing config
	# TODO: non-root: decline processing new config
	# as we are getting diff from the default config, we need to restore defaults first
	handle_me $g_loglvl_info "New config received. Applying defaults and received config on top of that."
	defaults
	# now apply the received configuration
	if echo "$g_sect_content" >"$g_floc_cfgimport" ; then
		if restore "$g_floc_cfgimport" ; then
			handle_me $g_loglvl_warn "Router configuration was updated, reboot operation pending..." "$g_floc_glog"
			g_todo_restart="yes, please"
		else
			handle_me $g_loglvl_warn "No configuration change when updating router configuration." "$g_floc_glog"
		fi
	else
		handle_me $g_loglvl_err "Unable to save requested configuration changes to \"$g_floc_cfgimport\" file" $g_floc_glog
	fi
	g_cycle=$((g_num_smallcycles+1)) # run bigcycle next
	g_time_cycoverride=$g_ctime_atonce # run it immediately
	rm -f "$g_floc_oldcfgfile" 2>$g_fn_devnull # force re-sending configuration after receiving cfg update
}

prep_tunnel() {
	if [ $# -ne 3 ] ; then
		handle_me $g_loglvl_err "prep_tunnel() - need 3 parameters" "GLOG"
		return 1
	fi
	local l_key l_deviceport l_servport
	l_key="$1"
	l_deviceport="$2"
	l_servport="$3"
	echo "-----BEGIN OPENSSH PRIVATE KEY-----" >$g_floc_tunkey || return 1
	echo "$l_key" | sed 's/:/\n/g' >>$g_floc_tunkey || return 1
	echo "-----END OPENSSH PRIVATE KEY-----" >>$g_floc_tunkey || return 1
	chmod 600 $g_floc_tunkey || return 1
	cat >$g_floc_tunnscript <<EOF
$g_shellmagic
echo \$\$>$g_floc_tunnpidfile
s_consec_fail=0
s_consec_maxfail=5
s_earlyfail=90
s_exectime=\$(date +%s)
s_stoptime=\$s_exectime
s_slacktime=$g_time_tunnerr
s_stop=0
s_runssh=1
s_sshpid=0
s_procerr_file="$g_floc_wterrfile.processed"
isleep() {
    sleep \$1 &
    wait \$!
}
cleanup() {
    if [ 0 -ne "\$s_sshpid" ] ; then
        kill \$s_sshpid
        s_sshpid=0
    fi
    echo 0 >$g_floc_tunnpidfile
    if [ $# -eq 1 ] ; then
        exit 0
    else
        exit 1
    fi
}
trap cleanup HUP INT ABRT TERM QUIT TSTP USR1 USR2
rm ~/.ssh/known_hosts >/dev/null
logerr() {
    l_datetime=\$(date "+%F %T")
    for i in "$g_floc_logfile" "$g_floc_glog"; do
        echo "$g_txtll_err" "[" "\$l_datetime" "]" "Webtunnel: " "\$1" >>"\$i"
        [ -n "\$2" ] && cat "\$s_procerr_file" >>"\$i"
    done
}
proc_ssh_errs() {
    [ -s "$g_floc_wterrfile" ] && {
        rm "\$s_procerr_file"
        cat "$g_floc_wterrfile" | grep -v "Permanently added" >"\$s_procerr_file"
        [ -s "\$s_procerr_file" ] && logerr "Dumping SSH errors during webtunnel run:" "."
    }
    rm "\$s_procerr_file"
}
while [ \$s_stop -eq 0 ] ; do
    if [ 0 -ne "\$s_runssh" ] ; then
        logerr "Starting tunnel..."
        ssh -N -n -o PasswordAuthentication=no -o StrictHostKeyChecking=no -o ExitOnForwardFailure=yes -o ServerAliveInterval=60 -o ServerAliveCountMax=3 -R $l_deviceport:127.0.0.1:443 -i $g_floc_tunkey -p $l_servport dwarfg@$g_serverurl 2>"$g_floc_wterrfile" &
        s_sshpid="\$!"
        s_runssh=0
        s_exectime=\$(date +%s)
    fi
    isleep \$s_slacktime
    if [ 0 -eq "\$s_sshpid" ] ; then
        s_stop=1
        logerr "Tunnel setup failed. Dismantling tunnel."
        proc_ssh_errs
        continue
    fi
    if [ ! -d "/proc/\$s_sshpid" ] ; then
        s_runssh=1
        s_stoptime=\$(date +%s)
        s_timediff=\$((s_stoptime-s_exectime))
        if [ \$s_timediff -lt \$s_earlyfail ] ; then
            s_consec_fail=\$((s_consec_fail+1))
            if [ \$s_consec_fail -gt \$s_consec_maxfail ] ; then
                logerr "Too many failures, dismantling tunnel."
                s_stop=1
            else
                isleep \$s_slacktime
            fi
        else
            s_consec_fail=0
        fi
        logerr "Tunnel (ssh) exited."
    fi
    proc_ssh_errs
done
proc_ssh_errs
cleanup "ok"
EOF
	[ $? -eq 0 ] || return 1
	chmod a+x $g_floc_tunnscript || return 1
	return 0
}

get_tunnel_pid() {
	local l_mytpid l_mcmd
	if [ -f "$g_floc_tunnpidfile" ] ; then
		read -r l_mytpid <"$g_floc_tunnpidfile"
		if [[ -n "$l_mytpid" && "0" != "$l_mytpid" && "$l_mytpid" = "${l_mytpid//[^0-9]}" && -d "/proc/$l_mytpid" ]] ; then
			if [ $# -eq 0 ] ; then
				echo "$l_mytpid"
				return 0
			else
				read -r l_mcmd <"/proc/$l_mytpid/comm"
				if [[ -n "$l_mcmd" && "$g_fn_tunnscript" = "$l_mcmd" ]] ; then
					echo "$l_mytpid"
					return 0
				fi
			fi
		fi
	fi
	echo "0"
}

start_tunnel() {
	local l_tpid
	l_tpid=0
	l_tpid=$(get_tunnel_pid)
	if [ "$l_tpid" -ne 0 ] ; then
		g_tunstat="$g_tuns_already"
		handle_me $g_loglvl_warn "Not starting tunnel - running already"
		return 0
	else
		[ -f "$g_floc_tunnscript" ] || {
			handle_me $g_loglvl_err "Tunnel script does not exist on tunnel start request"
			return 1
		}
		nohup $g_floc_tunnscript &
		[ $? -eq 0 ] || return 1
		isleep 10
		l_tpid=$(get_tunnel_pid)
		if [ -d "/proc/$l_tpid" ] ; then
			g_tunstat=$g_tuns_started
			handle_me $g_loglvl_warn "Started webtunnel script" "GLOG"
			return 0
		else
			g_tunstat=$g_tuns_error
			handle_me $g_loglvl_err "Webtunnel script start failed." "GLOG"
			return 1
		fi
	fi
}

stop_webtunnel() {
	local l_tunpid
	l_tunpid=$(get_tunnel_pid)
	if [ "0" -ne "$l_tunpid" ] ; then
		kill "$l_tunpid"
		g_tunstat=$g_tuns_stopped
		handle_me $g_loglvl_warn "Stoppped webtunnel" "GLOG"
	else
		[[ "$g_tunstat" = "$g_tuns_started" || "$g_tunstat" = "$g_tuns_already" ]] && {
			g_tunstat=$g_tuns_stopped
			handle_me $g_loglvl_warn "Webtunnel stop after tunnel defunct (ok)." "GLOG"
		}
	fi
}

process_commands() {
	local l_number l_cmd l_arg l_cmdline l_nospace l_upgpackage l_upgbasename l_upgdir l_url l_oldpwd l_servport l_deviceport l_key l_tunpid l_argrest l_rcc
	while IFS= read -r l_cmdline <&3
	do
		l_cmd=${l_cmdline%% *} # cut at first space
		l_arg=${l_cmdline#* } # cut at first space
		case $l_cmd in
			"DELAY")
				l_number="${l_arg//[^0-9]}"
				if [ "$l_arg" = "$l_number" ] ; then
					isleep "$l_arg"
					return 0
				else
					handle_me $g_loglvl_err "Not processing app DELAY command - arg is NaN (\"$l_arg\")"
				fi
				;;
			"HNAME")
				# TODO non-root - decline hostname
				l_nospace="${l_arg//[ ]}"
				if [ "$l_arg" = "$l_nospace" ] ; then
					hostname "$l_arg"
					return 0
				else
					handle_me $g_loglvl_err "Not processing app HNAME command - arg contains space (\"$l_arg\")"
				fi
				;;
			"UPGRADE")
				# TODO non-root: decline UPGRADE; NOTE: this covers agent upgrade, *not* FW UPDATE
				l_upgpackage="$l_arg"
				cd "$g_floc_appdir" || {
					handle_me $g_loglvl_err "Unable to change directory to $g_floc_appdir"
					return 1
				}
				handle_me $g_loglvl_dbg "Agent upgrade command is curl -s -S $g_curlsopt_var -F \"$g_form_reqfile=$l_upgpackage\" -F \"$g_form_devid=$g_devid\" -F \"$g_form_machid=$g_machid\" -F \"$g_form_devtoken=$g_device_token\" -F \"$g_form_servid=$g_servid\" -o \"$l_upgpackage\" \"$g_url_fileget\""
				if curl -s -S $g_curlsopt_var -F "$g_form_reqfile=$l_upgpackage" -F "$g_form_devid=$g_devid" -F "$g_form_machid=$g_machid" -F "$g_form_devtoken=$g_device_token" -F "$g_form_servid=$g_servid" -o "$l_upgpackage" "$g_url_fileget" ; then
					l_upgbasename=${l_upgpackage##*/}
					l_upgdir=${l_upgbasename%.tgz}
					if [ -d "$l_upgdir" ] ; then
						rm -rf "$l_upgdir" || {
							handle_me $g_loglvl_err "Unable to clean up upgrade directory $l_upgdir"
							return 1
						}
					fi
					if [ -d "$g_fn_tgtdir" ] ; then
						rm -rf "$g_fn_tgtdir" || {
							handle_me $g_loglvl_err "Unable to cleanup upgrade target directory $g_fn_tgtdir"
							return 1
						}
					fi
					tar xzf "$l_upgbasename" || {
						handle_me $g_loglvl_err "Unable to unpack agent upgrade archive $l_upgbasename"
						return 1
					}
					if [ "$l_upgdir" != "$g_fn_tgtdir" ] ; then
						mv "$l_upgdir" "$g_fn_tgtdir" || {
							handle_me $g_loglvl_err "Failed to rename upgrade directory $l_upgdir to upgrade target $g_fn_tgtdir"
							return 1
						}
					fi
					if [ ! -f "$g_fn_tgtdir/$g_fn_upgtgt" ] ; then
						mv "$g_fn_tgtdir/$g_fn_bintarget" "$g_fn_tgtdir/$g_fn_upgtgt" || {
							handle_me $g_loglvl_err "Failed to prepare $g_fn_upgtgt"
							return 1
						}
					fi
					[ -e "$g_floc_monupgexit" ] && rm "$g_floc_monupgexit" >$g_fn_devnull
					if command -v nohup ; then
						nohup "$g_fn_tgtdir/$g_fn_upgtgt" >/dev/null 2>&1 </dev/null &
						l_rcc=$?
					else
						"$g_fn_tgtdir/$g_fn_upgtgt" >/dev/null 2>&1 </dev/null &
						l_rcc=$?
					fi
					if [ $l_rcc -eq 0 ] ; then
						handle_me $g_loglvl_warn "Attempting Dwarfguard agent upgrade..." "$g_floc_glog"
						[ -n "$g_i_am_slave" ] && exit $g_retcode_upgrade # exit when being monitored
						process_cleanup "Old agent run: upgrade process executed, exiting" $g_retcode_upgrade # cleanup otherwise
					else
						handle_me $g_loglvl_err "Failed to start upgrade by running $g_fn_tgtdir/$g_fn_upgtgt" "$g_floc_glog"
						[ -n "$g_i_am_slave" ] && exit 1 # exit when there is a monitor
						return 1 # no monitor - just return an error
					fi
				else
					handle_me $g_loglvl_err "Error when downloading Dwarfguard agent package..."
				fi
				;;
			"UPDATE")
				# TODO non-root: decline UPDATE ; NOTE: this is FW UPDATE, *not* agent upgrade
				case $g_mysystem in
					"$g_sys_icros")
						l_nospace="${l_arg//[ ]}"
						if [ -z "$l_arg" ] ; then
							handle_me $g_loglvl_err "Garbled update command - no argument."
						elif [ "$l_arg" != "$l_nospace" ] ; then
							handle_me $g_loglvl_err "Garbled update command - argument contains spaces."
						else
							handle_me $g_loglvl_note "Processing FW update command with argument \"$l_arg\""
						fi
						if [ -z "$g_fwfname" ] ; then
							handle_me $g_loglvl_err "Skipping ordered FW upgrade - my FW file name is empty"
						fi
						l_url="$g_url_fwdown_conelos/$l_arg/$g_fwfname"
						[ "${l_url:(-4)}" != ".bin" ] && l_url="${l_url}.bin" # patch with .bin if missing
						l_oldpwd=$(pwd)
						l_upgdir="/tmp/dwarfg_fwupd_$$"
						mkdir "$l_upgdir" && cd "$l_upgdir" || { handle_me $g_loglvl_err "Unable to create and change to directory $l_upgdir" ; return 1 ; }
						if [ -f "$g_fwfname" ] ; then
							handle_me $g_loglvl_note "Removing old FW upgrade file."
							rm "$g_fwfname"
						fi
						curl -s -S $g_curlsopt_var "$l_url" -o "$g_fwfname"
						process_curl_response $? || {
							handle_me $g_loglvl_err "Error downloading new FW from URL: $l_url"
							cd "$l_oldpwd" && rm -rf "$l_upgdir"
							return 1
						}
						g_defs_sent=0 # flash to potentially new FW, need to re-send defaults
						save_settings "file" # remember updated settings
						handle_me $g_loglvl_warn "Performing FW flash to version $l_arg..." "$g_floc_glog"
						# first copy busybox or reboot into /tmp - in case of using fwupdate -n and logging output and result...
						# [ -f /tmp/reboot ] && rm /tmp/reboot
						# cp /usr/bin/cbox /tmp/reboot
						# real update execution
						fwupdate -i "$g_fwfname"
						# opt: fwupdate -i <file> -n >/root/fwup.log 2>/root/fwuperr.log - check if possible
						# opt: check return code, try to write it and pick it up on next boot to report results...
						# opt: /tmp/reboot
						# TODO: pick it up anyway, we do not neet to use -n for that
						cd "$l_oldpwd"
						return 0
						;;
					*)
						handle_me $g_loglvl_warn "Device-type does not support this command: $l_cmd"
						;;
				esac
				;;
			"CONFIG")
				# TODO non-root: decline CONFIG
				case $g_mysystem in
					"$g_sys_icros")
						l_oldpwd=$(pwd)
						l_url="$g_srvbase/config/${g_devid}.cfg"
						cd "$g_floc_appdir" || { handle_me $g_loglvl_err "Unable to change directory to $g_floc_appdir" ; return 1 ; }
						curl -s -S $g_curlsopt_var "$l_url" -o "$g_fn_newcfg"
						process_curl_response $? || {
							cd "$l_oldpwd"
							handle_me $g_loglvl_err "Error downloading new configuration from URL: $l_url"
							return 1
						}
						handle_me $g_loglvl_warn "Performing configuration update..." "$g_floc_glog"
						if restore $g_fn_newcfg ; then
							handle_me $g_loglvl_warn "Router configuration was updated, reboot operation pending..." "$g_floc_glog"
							g_todo_restart="yes, please"
						else
							handle_me $g_loglvl_warn "No configuration change when updating router configuration." "$g_floc_glog"
						fi
						cd "$l_oldpwd"
						return 0
						;;
					*)
						handle_me $g_loglvl_warn "Config update (command $l_cmd) not yet supported for this device type."
						;;
				esac
				;;
			# EXEC)
				# ;;
			"WEBTUNNEL")
				l_servport=${l_arg%% *} # cut at first space - first argument is server port
				l_argrest=${l_arg#* } # cut at first space - rest of arguments
				l_deviceport=${l_argrest%% *} # cut second argument - device port on server
				l_key=${l_argrest#* } # cut rest of arguments (third argument) - that is a ssh key
				l_tunpid=$(get_tunnel_pid)
				if [ "0" -ne "$l_tunpid" ] ; then
					g_tunstat=$g_tuns_already
				else
					if prep_tunnel "$l_key" "$l_deviceport" "$l_servport" ; then
							g_todo_tunnel="yes please"
					else
						handle_me $g_loglvl_err "Tunnel preparation failed, tunnel will NOT be started." "GLOG"
					fi
				fi
				save_settings "file"
				g_time_cycoverride=$g_ctime_atonce # one time immediate loop to server - result of web tunnel op
				return 0
				;;
			"WEBTUNNELSTOP")
				stop_webtunnel
				save_settings "file"
				g_time_cycoverride=$g_ctime_atonce # one time immediate loop to server - result of web tunnel op
				return 0
				;;
			"TIMESET")
				if [[ -z "$l_arg" || "$l_arg" != "${l_arg//[^0-9]}" ]] ; then
					handle_me $g_loglvl_err "TIMESET command argument contains garbage characters, ignoring..."
				else
					date -s "@$l_arg" && handle_me $g_loglvl_warn "Updated system time based on server request."
				fi
				;;
			"REBOOT")
				handle_me $g_loglvl_warn "Server requested reboot, rebooting..." $g_floc_glog
				reboot
				;;
			*)
				handle_me $g_loglvl_err "Unsupported command from app: \"$l_cmd\""
		esac
	done 3<<EOF
$(echo -e "$g_sect_content")
EOF
	return 1 # only errors here
}

process_custom_scripts() {
	[ -d $g_floc_csdir ] || mkdir -p $g_floc_csdir || {
		handle_me $g_loglvl_err "Unable to create custom scripts dir ($g_floc_csdir)"
		return 1
	}
	while IFS= read -r l_cmdline <&3
	do
		l_cmd=${l_cmdline%% *} # cut at first space
		l_arg=${l_cmdline#* } # cut at first space
		if [ -n "$l_arg" ] ; then
			ftodo="$g_floc_csdir/$g_ctxt_csprefix$l_arg"
			fexist=""
			[ -f "$ftodo" ] && fexist="y"
			case $l_cmd in
				"CUSTSCRIPT_ADD")
					# curl -s -S $g_curlsopt -o "file"
					handle_me $g_loglvl_dbg "Custscript download command is curl -s -S $g_curlsopt_var -F \"$g_form_reqfile=$g_ctxt_csprefix$l_arg\" -F \"$g_form_devid=$g_devid\" -F \"$g_form_machid=$g_machid\" -F \"$g_form_devtoken=$g_device_token\" -F \"$g_form_servid=$g_servid\" -o \"$ftodo\" \"$g_url_fileget\""
					if curl -s -S $g_curlsopt_var -F "$g_form_reqfile=$g_ctxt_csprefix$l_arg" -F "$g_form_devid=$g_devid" -F "$g_form_machid=$g_machid" -F "$g_form_devtoken=$g_device_token" -F "$g_form_servid=$g_servid" -o "$ftodo" "$g_url_fileget" ; then
						chmod a+x "$ftodo"
						if [ -n "$fexist" ] ; then
							handle_me $g_loglvl_warn "Upgraded custom script $ftodo"
						else
							handle_me $g_loglvl_warn "Downloaded new custom script $ftodo"
						fi
					fi
					;;
				"CUSTSCRIPT_DEL")
					words=$(echo "$ftodo" | wc -w)
					[ 1 -ne $words ] && {
						handle_me $g_loglvl_err "Custom script filename ($ftodo) contains spaces, skipping..."
						continue
					}
					rm ${ftodo}[0-9]* || {
						handle_me $g_loglvl_err "Failed to delete customscript \"${ftodo}[0-9]*\""
						continue
					}
					handle_me $g_loglvl_warn "Deleted custom script $ftodo"
					;;
				*)
					handle_me $g_loglvl_err "Unsupported custom script directive from app: \"$l_cmd\""
			esac
		else
			handle_me $g_loglvl_err "Empty customscript name / custscript name parse error from: \"$l_cmdline\""
		fi
	done 3<<EOF
$(echo -e "$g_sect_content")
EOF
	return 1
}

add_defaults_script() {
	local l_begstr l_fname l_line
	[ $# -ne 2 ] && handle_me $g_loglvl_err "Invalid use of add_defaults_script" && return
	l_begstr="$1"
	l_fname="$2"
	[ ! -f "$l_fname" ] && return
	while IFS= read -r l_line
	do
		echo "${l_begstr}=$l_line" >>$g_floc_gdatadir/$g_fn_defmerge
	done <"$l_fname"
}

get_defaults() {
	case $g_mysystem in
		"$g_sys_icros")
			rm -f "$g_floc_gdatadir/$g_fn_defmerge" 2>$g_fn_devnull
			for i in /etc/defaults/settings*; do
				cat "$i" >>"$g_floc_gdatadir/$g_fn_defmerge"
			done
			add_defaults_script "STARTUP" /etc/defaults/rc.d/rc.local
			add_defaults_script "IP_UP" /etc/defaults/scripts/ip-up.local
			add_defaults_script "IP_DOWN" /etc/defaults/scripts/ip-down.local
			add_defaults_script "IP6_UP" /etc/defaults/scripts/ip6-up.local
			add_defaults_script "IP6_DOWN" /etc/defaults/scripts/ip6-down.local
			tar czf "$g_floc_gdatadir/$g_fn_defaults" "$g_floc_gdatadir/$g_fn_defmerge"
			;;
		"$g_sys_linux")
			tar czf "$g_floc_gdatadir/$g_fn_defaults" /etc/default
			;;
		"$g_sys_owrt")
			: # TODO where are defaults on OpenWRT?
			;;
		"$g_sys_teltonika")
			: # TODO where are defaults on RutOS?
			;;
		"$g_sys_cstech")
			: # no defaults on CS-Tech device
			;;
	esac
}

run_small() {
	local l_output l_linecount l_ecnt l_cust_vars l_cust_res
	handle_me $g_loglvl_dbg "run_small()"
	update_machid
	l_ecnt=$g_ecnt_small
	l_output=$($g_stat_small)
	l_linecount=$(echo -ne "$l_output"|grep -c "^")
	if [ 0 -eq "$l_linecount" ] ; then
		handle_me $g_loglvl_err "run_small produced empty output, discarding this cycle output"
		return 1
	elif [ "$g_ecnt_small" -gt "$l_ecnt" ] ; then
		handle_me $g_loglvl_err "run_small errors: $((g_ecnt_small-l_ecnt)) (out of $g_ecnt_small total)"
	fi
	g_txtbuff_incr=""
	l_cust_res=""
	[ -z "$g_custom_scripts_disabled" ] && {
		l_cust_vars=$(run_custscripts)
		add_sect_tobuff "$g_cds_sname_cust" "$l_cust_vars"
		l_cust_res="$g_txtbuff_incr"
	}
	if [ -n "$l_cust_res" ] ; then
		echo -n "$l_output\n$l_cust_res"
	else
		echo -n "$l_output"
	fi
	return 0
}

run_big() {
	local l_ecnt l_output l_linecount l_cust_res
	handle_me $g_loglvl_info "run_big()"
	l_ecnt=$g_ecnt_big
	l_output=$($g_stat_big)
	l_linecount=$(echo -ne "$l_output"|grep -c "^")
	handle_me $g_loglvl_dbg "l_linecount is \"$l_linecount\""
	if [ 0 -eq "$l_linecount" ] ; then
		handle_me $g_loglvl_err "run_big produced empty output, discarding this cycle output"
		return 1
	elif [ "$g_ecnt_big" -gt "$l_ecnt" ] ; then
		handle_me $g_loglvl_err "run_big errors: $((g_ecnt_big-l_ecnt)) (out of $g_ecnt_big total)"
	fi
	g_txtbuff_incr=""
	l_cust_res=""
	[ -z "$g_custom_scripts_disabled" ] && {
		l_cust_vars=$(run_custscripts)
		add_sect_tobuff "$g_cds_sname_cust" "$l_cust_vars"
		l_cust_res="$g_txtbuff_incr"
	}
	if [ -n "$l_cust_res" ] ; then
		echo -n "$l_output\n$l_cust_res"
	else
		echo -n "$l_output"
	fi
	return 0
}

cert_update() {
	local l_server_response l_tmpline l_linecount l_lc_head l_lc_tail l_respcode l_retcode l_server_id
	[ "$g_security" -eq 2 ] && {
		handle_me $g_loglvl_warn "Updating certificate on security level $g_security is not possible" "$g_floc_glog"
		isleep "$g_time_certupderr"
		return 1
	}
	[[ "$g_security" -eq 1 && -z "$g_cert_token" ]] && {
		handle_me $g_loglvl_warn "Unable to update certificate - empty cert token."
		isleep "$g_time_certupderr"
		return 1
	}
	habdle_me $g_loglvl_dbg "Cert update command is curl $g_cursopt_nocheck -s -W -w \"\n%{http_code}\" -F \"$g_form_devid=$g_devid\" -F \"$g_form_machid=$g_machid\" -F \"$g_form_protover=$g_protoversion\" -F \"$g_form_servid=$g_servid\" \"$g_url_certupd\""
	if l_server_response=$(curl $g_cursopt_nocheck -s -W -w "\n%{http_code}" -F "$g_form_devid=$g_devid" -F "$g_form_machid=$g_machid" -F "$g_form_protover=$g_protoversion" -F "$g_form_servid=$g_servid" "$g_url_certupd") ; then
		# TODO: respect DELAY_TIME if issued by server
		l_linecount=$(echo -ne "$l_server_response" | grep -c "^")
		l_respcode=$(echo -ne "$l_server_response"|tail -1)
		case $l_respcode in
			''|*[!0-9]*)
				handle_me $g_loglvl_err "Empty or NaN response code from server during certificate refresh (\"$l_respcode\")"
				return 1
				;;
			200)
				l_tmpline=$(echo -ne "$l_server_response"|head -1)
				l_retcode=${l_tmpline%%${g_ctxt_devidsep}*} # strip end (longest) (code:whatever:servid:sleeptime?)
				l_server_id=${l_tmpline#*${g_ctxt_devidsep}} # remove ret code from servid
				l_server_id=${l_server_id%%${g_ctxt_devidsep}*} # polish servid in case anything follows
				if [ "$l_server_id" != "$g_servid" ] ; then
					handle_me $g_loglvl_err "Error: Different server ID when replacing cetificate."
					return 1
				fi
				case $l_retcode in
					''|*[!0-9]*)
						handle_me $g_loglvl_err "Invalid application return code from server on cert refresh."
						return 1
						;;
					"$g_aerr_invid")
						handle_me $g_loglvl_err "Server does not know this device - no cert refresh possible."
						return 1
						;;
					"$g_aerr_notex")
						handle_me $g_loglvl_err "Server does not manage its certificate - no cert refresh possible."
						return 1
						;;
					"$g_aerr_invda")
						handle_me $g_loglvl_err "Server responds with Invalid data (device token corrupted?)"
						return 1
						;;
					"$g_aerr_neser")
						handle_me $g_loglvl_err "Improper server contacted / server uses different Server ID than stored in this agent. Fix server URL or update agent."
						return 1
						;;
					0)
						l_lc_head=$((l_linecount-1))
						l_lc_tail=$((l_lc_head-1))
						if [ "$l_lc_tail" -gt 0 ] ; then
							if echo -ne "$l_server_response" | head -$l_lc_head | tail -$l_lc_tail | openssl enc -d -aes-128-cbc -base64 -K "$g_device_token" -iv "$g_cert_token" >"$g_floc_appdir/$g_fn_newcert"; then
								mv "$g_floc_appdir/$g_fn_newcert" "$g_floc_appdir/$g_fn_cert"
								handle_me $g_loglvl_warn "Server certificate refreshed." "$GLOG"
								return 0
							else
								handle_me $g_loglvl_err "Error $? when decoding certificate." "$GLOG"
								return 1
							fi
						else
							handle_me $g_loglvl_err "Server returned empty response on cert refresh."
						fi
						;;
					*)
						handle_me $g_loglvl_err "Unknown app response from server ($l_retcode) during cert-refresh."
						return 1
						;;
				esac
				;;
			*)
				handle_me $g_loglvl_err "HTTP Error code $l_respcode received when refreshing server certificate."
				return 1
		esac
	else
		handle_me $g_loglvl_err "Error $? when attempting to refresh certificate from server."
	fi
}

detect_system() {
	[ $# -eq 1 ] && unset g_mysystem # force re-detect when parameter provided
	if [ -z "$g_mysystem" ] ; then
		if status lan >$g_fn_devnull 2>&1 ; then
			if [[ -e "/sys/hal" && -e "/etc/cudo.conf" ]] ; then
				g_mysystem="$g_sys_cstech"
			else
				g_mysystem="$g_sys_icros"
			fi
		elif [ -f /etc/issue ] ; then
			g_mysystem="$g_sys_linux"
		elif [ -f /etc/banner ] && grep -i "openwrt" /etc/banner ; then
			g_mysystem="$g_sys_owrt"
			g_shellmagic="#!/bin/sh"
			g_initmagic='/etc/rc.common

START=90
STOP=10'
			g_ctxt_initbody=""
			g_ctxt_matchdetect='while read -r -d "" l_myvar; do [ "${l_myvar##*/}" = "$g_fn_bintarget" ] && l_match="yes" && break ; done <"/proc/$input/cmdline"'
		elif [ -f /etc/banner ] && grep -i "teltonika" /etc/banner ; then
			g_mysystem="$g_sys_teltonika"
			g_shellmagic="#!/bin/sh"
			g_initmagic='/etc/rc.common

START=90
STOP=10'
			g_ctxt_initbody=""
			g_ctxt_matchdetect='x_input=$(cat /proc/$input/cmdline)
					x_input="${x_input##*/}"
					[ "${x_input%% *}" = "$g_fn_bintarget" ] && l_match="yes" && break'
		else
			g_mysystem="$g_sys_other"
		fi
	fi
}

# sets global variable g_otherpid to match PID of Dwarfguard monitor. set to 0 if not detected
detect_running() {
	local l_input l_inpid l_match l_realbin l_nowrite
	l_input=""
	l_inpid="$g_otherpid"
	l_nowrite="$1"
	[ -f "$g_floc_pidfile" ] && { # pidfile is there, read pid from it
		read -r l_input<$g_floc_pidfile
		case "$l_input" in
			'') # empty line
				handle_me $g_loglvl_err "Pidfile $g_floc_pidfile does not contain adwarfg PID." "$g_floc_glog"
				;;
			*[!0-9]*) # contains a non-number character
				handle_me $g_loglvl_err "Pidfile $g_floc_pidfile contains non-number characters." "$g_floc_glog"
				;;
			*) # otherwise it is a number, update vars
				g_otherpid="$l_input" && l_inpid="$l_input"
		esac
	}
	if [ "$g_otherpid" -eq $$ ] ; then
		g_otherpid=0 # BINGO - got same PID as previous agent (actually quite possible on ICR OS)
	elif [ 0 -ne "$g_otherpid" ] ; then
		# supposed to be running
		if [ -d "/proc/$g_otherpid" ] ; then
			l_match=""
			l_realbin=$(basename "$g_fn_bintarget")
			if [ "$g_mysystem" = "$g_sys_teltonika" ] ; then # the while read below does not work on RutOS (though it works on OpenWRT)
				l_input=$(cat /proc/$g_otherpid/cmdline)
				l_input="${l_input##*/}"
				[ "${l_input%% *}" = "$l_realbin" ] && l_match="yes"
			else
				while read -r -d $'\0' l_input; do # read parts of /proc/PID/cmdline
					[ "${l_input##*/}" = "$l_realbin" ] && l_match="yes" && break
				done <"/proc/$g_otherpid/cmdline"
			fi
			# reset pid to zero if not adwarfg process
			[ -z "$l_match" ] && handle_me $g_loglvl_err "Supposed adwarfg monitor PID $g_otherpid is actually not a $g_fn_bintarget process ($l_realbin), cleaning up... [from: $1]" "$g_floc_glog" && g_otherpid=0
		else # not running
			g_otherpid=0
		fi
	fi
	[[ "$g_otherpid" -ne "$l_inpid" && -z "$l_nowrite" ]] && echo "$g_otherpid" >$g_floc_pidfile # update pidfile if real pid differs from the stored one
}

write_pidfile() {
	local l_result
	l_result=1
	[ 0 -eq "$g_otherpid" ] || handle_me $g_loglvl_err "Overwriting PIDfile when other agent process ($g_otherpid) was detected." $g_floc_glog
	if echo $$>$g_floc_pidfile ; then
		l_result=0
	else
		handle_me $g_loglvl_err "Unable to store my PID in g_floc_pidfile ($g_floc_pidfile). Expect problems." $g_floc_glog
	fi
	return $l_result
}

process_parameters() {
	local l_par
	while [ $# -gt 0 ] ; do
		case "$1" in
			"$g_ipar_expedite")
				expedite
				exit 0
				;;
			"$g_ipar_keeptalking")
				g_kill_output=""
				;;
			"$g_ipar_monitored"|"$g_ipar_nomon")
				g_i_am_slave="yes please"
				;;
			"-h"|"-help"|"--help")
				echo -e "$g_ctxt_help"
				exit
				;;
			"-v"|"-version"|"--version")
				echo $g_appversion
				exit
				;;
			"-D"|"-DEBUG"|"--DEBUG")
				g_par_setdebug="yes"
				g_par_setverbose="yes"
				g_child_params="$g_child_params $1"
				g_i_am_slave="yes please"
				g_kill_output=""
				;;
			"-d"|"-debug"|"--debug")
				g_par_setverbose="yes"
				g_child_params="$g_child_params $1"
				g_i_am_slave="yes please"
				g_kill_output=""
				;;
			"-c")
				echo "Settings RESET requested. NOTE: settings are backed up."
				save_settings "RESET"
				exit $?
				;;
			"-cache")
				g_par_cache="yes please"
				g_child_params="$g_child_params $1"
				;;
			"-r")
				echo "Settings RESTORE requested. NOTE: (potentially) existing settings are backed up (to a file: $g_floc_resetbcksettings)"
				save_settings "RESTORE"
				exit $?
				;;
			"-o")
				g_num_loopend=1
				g_i_am_slave="yes please"
				;;
			"-omax")
				l_par=$2
				g_i_am_slave="yes please"
				if [[ -n "$2" && "$2" = "${param//[^0-9]}" ]] ; then
					g_num_loopend=$l_par
					g_child_params="$g_child_params $1 $2"
					shift
				else
					echo "Number of loops omitted, going for one loop only." >&2
					g_num_loopend=1
					g_child_params="$g_child_params $1"
				fi
				;;
		esac
		shift
	done
}

cleanup_pending_synctime() {
	[ $# -eq 0 ] && echo "time_target=" >"$g_floc_webdata" # clean up any pending sync time
}

prep_logfile() { # call with any parameter to play it safe (no disruption to other potentially running agents)
	local l_skip_local_log="$1"
	[ -d $g_floc_gdatadir ] || mkdir -p $g_floc_gdatadir || handle_me $g_loglvl_fatal "Unable to create datadir ($g_floc_gdatadir)"
	# > sort out global log file
	if [ ! -f $g_floc_glog ] ; then touch $g_floc_glog; fi # create global log file if it does not exist
	# > hail the world in global log file (works only if lower debug mode is set globally)
	handle_me $g_loglvl_note "$g_txtappname_long version $g_appversion (mode: $g_mode) starting up PID $$..." "$g_floc_glog"
	# > create my log file
	[ -z "$l_skip_local_log" ] && {
		mkdir -p "$g_floc_mydir" || handle_me $g_loglvl_fatal "Unable to create mydir ($g_floc_mydir)." "$g_floc_glog"
		[ -f "$g_floc_logfile" ] && {
			rm "$g_floc_logfile" || {
				echo "Testing append from new agent..." >>$g_floc_logfile || {
					handle_me $g_loglvl_err "Old logfile existing and cannot be removed or appended." "$g_floc_glog"
					return 1
				}
				handle_me $g_loglvl_warn "Reusing old logfile - appeding from new agent..." "$g_floc_glog"
			}
		}
		touch $g_floc_logfile || {
			handle_me $g_loglvl_err "Unable to create logfile ($g_floc_logfile.), running without log..." "$g_floc_glog"
			return 1
		}
		# > hail the world in our log file
		handle_me $g_loglvl_warn "$g_txtappname_long version $g_appversion (mode: $g_mode) starting up PID $$..."
	}
	return 0
}

fix_min_date() {
	local heredate
	heredate=$(date "+%F")
	[ -z "$g_fix_mindate" ] && return 1
	[ -z "$heredate" ] && {
		handle_me $g_loglvl_warn "Invalid date returned from date command."
		return 1
	}
	[ "$g_fix_mindate" \> "$heredate" ] && {
		handle_me $g_loglvl_warn "System date too old. Attempting to set to $g_fix_mindate..."
		date -s "$g_fix_mindate $(date "+%T")"
	}
}

trap '' HUP
trap 'process_cleanup terminated/kill 1' TERM
trap 'process_cleanup terminated/terminal 1' INT TSTP QUIT
trap 'process_cleanup terminated/usrX 1' USR1 USR2
trap 'process_cleanup aborted 1' ABRT

# main program: init
detect_system "drop"
# 1st: check if deployed already and deploy if not
[[ -d "$g_floc_appdir" && -f "$g_floc_bintarget" ]] || {
	deploy
	m_cmdretcode=$?
	# now start in particular way depending on deploy retcode
	[ 0 -ne "$m_cmdretcode" ] && {
		g_deployment="$m_cmdretcode"
		save_settings "file"
	}
	case "$m_cmdretcode" in
		"$g_deployd_systemd")
			echo "Dwarfguard Agent was installed and enabled within systemd, starting it up now..."
			systemctl start adwarfg.service || handle_me $g_loglvl_fatal "Failed to start adwarfg via systemd."
			;;
		"$g_deployd_sysv")
			$g_fn_sysvdir/adwarfg start || handle_me $g_loglvl_fatal "Failed to start adwarfg via initd."
			;;
		"$g_deployd_own")
			$g_floc_bintarget &
			[ $? -eq 0 ] || handle_me $g_loglvl_fatal "Failed to start adwarfg (no integration)."
			;;
		*)
			handle_me $g_loglvl_fatal "Deployment error, agent is not started up."
			;;
	esac
	exit 0
}
prep_logfile ""
fix_min_date
detect_running ""
# 2st: run upgrade if called as upgrade script
[ "$g_fn_upgtgt" = "${0##*/}" ] && { # test for upgrade first
	[ ! -d "$g_floc_appdir" ] && echo "Agent is not installed, nothing to upgrade ($g_floc_appdir is missing). Rename me to $g_fn_bintarget and simply execute as root to install agent. It will install the current version (no need to run upgrade afterwards)." >&2 && exit 1
	g_mode=$g_cmode_upgrade
	handle_me $g_loglvl_warn "Switching mode to $g_mode."
	touch "$g_floc_upgrading"
	g_deployment=$(grep "^g_deployment=" "$g_floc_bintarget" | sed "s/.*=//")
	[ -z "$g_deployment" ] && handle_me $g_loglvl_fatal "Failed to detect deployment type for adwarfg." "$g_floc_glog"
	agent_upgrade
	m_res=$?
	rm "$g_floc_upgrading" || m_res=2
	exit $m_res
}
# wait if upgrade is running
i=0
while [ $i -lt 5 ] ; do
	[ ! -e "$g_floc_upgrading" ] && break
	sleep 1
	i=$((i+1))
done
# still upgrading? should really be a stale lockfile...
[ -e "$g_floc_upgrading" ] && {
	rm "$g_floc_upgrading" >$g_fn_devnull
	handle_me $g_loglvl_err "Upgrade lockfile removed on startup, possibly relict of failed upgrade." $g_floc_glog
}
process_parameters "$@"
# 3nd: exit if other agent is running
[[ 0 -eq "$g_otherpid" || -n "$g_i_am_slave" ]] || {
	process_cleanup "Other agent ($g_otherpid) is running already, exiting" 0
}
# intermezzo: drop permissions if running with root and should not
# TODO  non-root
[ -z "$g_kill_output" ] || {
	exec >$g_fn_devnull
	exec 2>&1
}
load_settings
# 4th: run as monitor if required
# check if monitor needed - PID detected, systemd or monitored parameter => no monitor requested
#if [[ 0 -ne "$g_otherpid" || "$g_deployment" -eq "$g_deployd_systemd" || -n "$g_i_am_slave" ]] ; then
if [[ 0 -ne "$g_otherpid" || -n "$g_i_am_slave" ]] ; then
	g_txt_monitor=""
	[ -n "$g_i_am_slave" ] && {
		g_mode=$g_cmode_slave
		handle_me $g_loglvl_note "Switching mode to $g_mode." "$g_floc_glog"
	}
else
	g_txt_monitor="yes please"
	g_mode=$g_cmode_monitor
	handle_me $g_loglvl_info "Switching mode to $g_mode." "$g_floc_glog"
fi
m_dir=$(cd "${0%/*}" && pwd)
m_tdir=${g_floc_bintarget%/*}
[ "$m_tdir" != "$m_dir" ] && handle_me $g_loglvl_fatal "Running outside of target directory ($m_tdir) but not deploying is not allowed - did you forgot to remove previous agent installation? (call $g_floc_appdir/$g_fn_uninst)."
[ -n "$g_txt_monitor" ] && { # monitor mode
	l_num_failcounter=0
	[ -e "$g_floc_monupgexit" ] && rm "$g_floc_monupgexit" >$g_fn_devnull
	write_pidfile
	sleep 1
	g_childpid=0
	m_firstrun="yes"
	while true; do
		[[ $g_childpid -gt 0 && -d /proc/$g_childpid ]] || { # child running or exec it
			if [ -z "$m_firstrun" ] ; then
				wait $g_childpid
				m_childretcode=$?
				g_childpid=0 # child returned already so forget the pid
				case "$m_childretcode" in
					"$g_retcode_upgrade")
						# - stop or disable respawn based on deploy mode
						if [ "$g_deployment" = "$g_deployd_systemd" ] ; then
							handle_me $g_loglvl_warn "Monitor: upgrade under systemd, waitloop..." "$g_floc_glog"
							sleep 4
							m_counter=0
							while [[ -e "$g_floc_upgrading" && $m_counter -lt 3 ]] ; do
								m_counter=$((m_counter+1))
								sleep 2
							done
							[ -e "$g_floc_upgrading" ] && handle_me $g_loglvl_err "Monitor: upgrade lock existing after timeout, possible broken upgrade..." "$g_floc_glog"
							touch "$g_floc_monupgexit"
							process_cleanup "Monitor exiting because of agent upgrade" 0
						elif [ "$g_deployment" = "$g_deployd_sysv" ] ; then
							handle_me $g_loglvl_warn "Monitor: upgrade under SysV, exiting..." "$g_floc_glog"
						else
							handle_me $g_loglvl_warn "Monitor: upgrade under no integration, exiting..." "$g_floc_glog"
						fi
						touch "$g_floc_monupgexit"
						process_cleanup "Monitor exiting because of agent upgrade pending" 0
						;;
					*)
						l_num_failcounter=$((l_num_failcounter+1))
						;;
				esac
			else
				m_firstrun=""
			fi
			$g_floc_bintarget $g_ipar_monitored $g_child_params &
			g_childpid=$!
			handle_me $g_loglvl_warn "Dwarfguard monitor (PID $$): started new process $g_childpid" "$g_floc_glog"
			echo "$g_childpid" >"$g_floc_childpidfile" 2>$g_fn_devnull
		}
		[ $l_num_failcounter -gt $g_cnum_mfailinc ] && g_num_monsleep=$((g_num_monsleep+1)) && handle_me $g_loglvl_err "adwarfg failed too many times, increasing loop sleep to $g_num_monsleep." $g_floc_glog && l_num_failcounter=0
		isleep $g_num_monsleep
	done
	process_cleanup "Dwarfguard monitor exit." 0
}
# follows a non-monitor execution (single agent without monitor)
[ -z "$g_i_am_slave" ] && { # there is no monitor running and we are not monitor - we own the PIDfile
	write_pidfile
	sleep 1
}
# 5th: cleanup old directories (> 7 days old by default)
if [ "$g_mysystem" = "$g_sys_teltonika" ] ; then
	l_timestamp_file="/opt/adwarfg/find_timestamp"
	touch -d "@$(($(date +%s) - 7 * 86400))" "$l_timestamp_file"
	for i in $(find /opt/adwarfg/ -type d -iname "[0-9]*") ; do
		[[ -f "$i/logfile.txt" && "$i" -ot "$l_timestamp_file" ]] && rm -r "$i"
	done
else
	for i in $(find /opt/adwarfg/ -type d -iname "[0-9]*" -mtime +$g_logclean_days) ; do
		[ -f "$i/logfile.txt" ] && rm -r "$i"
	done
fi
handle_me $g_loglvl_warn "system is $g_mysystem" $g_floc_glog
l_savesystem="$g_mysystem" # save detected system
	# > load settings from file
load_settings
g_mysystem="$l_savesystem" # overwrite the (potentially wrong) system from settings.ini with what we detected on startup
# set tunnel status according to reality regardles on saved settings
m_tpid=$(get_tunnel_pid "startup")
if [ "$m_tpid" -ne 0 ] ; then
	g_tunstat=$g_tuns_started
	handle_me $g_loglvl_note "Found running tunnel on agent start."
else
	if [ "$g_tunstat" -ne "$g_tuns_none" ] ; then
		handle_me $g_loglvl_note "Tunnel no longer running on agent start (last saved state was $g_tunstat)."
		g_tunstat=$g_tuns_none
	fi
	echo "0" >$g_floc_tunnpidfile
fi
if [ -n "$g_todo_tunnel" ] ; then
	if [ "$g_tuns_started" -ne "$g_tunstat" ] ; then
		start_tunnel || handle_me $g_loglvl_err "Tunnel startup failed." "GLOG"
	fi
	g_todo_tunnel=""
	save_settings "file" # either started or running already or start failed -> handled/failed anyway
fi
# > change intervals if debug
if [[ -n "$m_debug" || -n "$g_par_setdebug" ]] ; then
	g_time_reg=$g_ctime_regdbg
	g_time_fall=$g_ctime_falldbg
	g_time_nextcyc=$g_ctime_nextdbg
	m_debug="yes"
fi
if [[ -n "$g_par_setverbose" ]] ; then
	g_alevel_debug=$g_loglvl_dbg
	g_alevel_log=$g_loglvl_dbg
	handle_me $g_loglvl_dbg "Running in DEBUG mode"
fi
[ -z "$g_par_setverbose" ] && save_settings "file" # if not debug then write out agent version in the settings
# > zero out and reset vars
g_cycle=0
g_ecnt_small=0
g_ecnt_big=0
g_connfail=0
g_sect_content=""
if [ "$g_alevel_log" -gt "$g_loglvl_fatal" ] ; then # fix log level (always process fatal failures) - DO NOT TOUCH THIS, can *CRIPPLE* program completely
	g_alevel_log=$g_loglvl_fatal
fi
# > determine client type and based on that determine some basic stuff
case $g_mysystem in
	"$g_sys_icros"|"$g_sys_cstech")
		g_stat_small=stat_small_advrouter
		g_stat_big=stat_big_advrouter
		g_cmd_ramd=$g_cmd_ramdrouter
		g_fwfname=$($g_cds_stat_sys -v | grep "^Product Name" | sed "s/^[^:]*:[ \t]*//")
		if [ -z "$g_fwfname" ] ; then
			g_fwfname="$g_fback_fwfnrouter"
		fi
		;;
	"$g_sys_linux")
		g_stat_small=stat_small_linux
		g_stat_big=stat_big_linux
		g_cmd_ramd=$g_cmd_ramdlinux
		g_fwfname="$g_fback_fwfnlinux"
		;;
	"$g_sys_owrt"|"$g_sys_teltonika")
		g_stat_small=stat_small_linux
		g_stat_big=stat_big_linux
		g_cmd_ramd=$g_cmd_ramdlinux
		g_fwfname="$g_fback_fwfnlinux"
		g_shellmagic="#!/bin/sh"
		;;
	*)
		handle_me $g_loglvl_fatal "System type ($g_mysystem) is not supported, bailing out."
esac
if [[ -z "$g_security" || "$g_security" != "${g_security//[^0-9]}" ]] ; then
	g_security=${g_security:-$g_defsecurity}
fi
case $g_security in
	"0")
		g_curlsopt_var="$g_cursopt_nocheck"
		;;
	*)
		if [ -r "$g_floc_appdir/$g_fn_cert" ] ; then
			g_curlsopt_var="$g_curlsopt_useloccert"
		else
			case $g_mysystem in
				"$g_sys_icros")
					if [ -x "$g_floc_certdir" ] ; then
						g_curlsopt_var="$g_curlsopt_def"
					else
						handle_me $g_loglvl_warn "Certs dir ($g_floc_certdir) not accessible, degrading security to 0 (accepting any server certificate)" $g_floc_glog
						g_curlsopt_var="$g_cursopt_nocheck"
					fi
				;;
			*)
				g_curlsopt_var="$g_curlsopt_def"
				;;
			esac
		fi
		;;
esac
# > hail the world in log file
handle_me $g_loglvl_note "$g_txtappname_long version $g_appversion is up, system is $g_mysystem"
if [ ! -d $g_floc_tmpdir ] ; then
	mkdir $g_floc_tmpdir || handle_me $g_loglvl_err "Unable to create $g_floc_tmpdir. Ramdisk will fail."
fi
grep $g_floc_tmpdir </proc/mounts >$g_fn_devnull 2>&1 && { # if g_floc_tmpdir not mounted, (try to) mount it
	handle_me $g_loglvl_info "Trying to mount ramdisk on $g_floc_tmpdir..."
	if $g_cmd_ramd ; then
		g_floc_smallcycerr="$g_floc_tmpdir/$g_fn_smallcycerr"
		g_floc_bigcycerr="$g_floc_tmpdir/$g_fn_bigcycerr"
	else
		handle_me $g_loglvl_warn "Unable to create ramdisk - errors during data gathering will be ignored."
	fi
}
[ -n "$g_bigcyclenext" ] && {
	g_bigcyclenext="" # disable the variable
	g_cycle=$((g_num_smallcycles+1)) # run bigcycle next
	save_settings "file" # save the disabled variable state into settings.ini
}
handle_me $g_loglvl_note "Server is $g_serverurl"
g_output=""
m_tunstat_cleanup=0
# main program: loop
cleanup_pending_synctime
while true; do
	g_cycle=$((g_cycle+1))
	[[ -n "$g_i_am_slave" && ! -d /proc/$PPID ]] && process_cleanup "Monitor exited, ending."
	if [ "$g_tunstat" -eq "$g_tuns_stopped" ] ; then
		m_tunstat_cleanup=1
	elif [[ "$g_tunstat" -eq "$g_tuns_started" || "$g_tunstat" -eq "$g_tuns_already" ]] ; then
		# check tunnel is still running...
		m_tpid=0
		if [ -f "$g_floc_tunnpidfile" ] ; then
			read -r m_tpid <$g_floc_tunnpidfile
		fi
		if [[ 0 -eq "$m_tpid" || ! -d "/proc/$m_tpid" ]] ; then
			g_tunstat="$g_tuns_stopped"
			save_settings "file"
		fi
	fi
	handle_me $g_loglvl_dbg "g_cycle is $g_cycle"
	# > sort out registration
	if [ -z "$g_devid" ] ; then
		if device_register ; then
			:
		else
			handle_me $g_loglvl_warn "Device registration failed, looping..."
			g_time_nextcyc=$g_time_fall
			isleep $g_time_nextcyc
			continue
		fi
	fi
	if [[ 0 -ne $g_connfail && $g_num_conntol -gt $g_connfail ]] ; then # after conenction fail but in tolerance (data still fresh)
		handle_me $g_loglvl_info "Skipping data creation because of comm issues (skipped ${g_connfail}/$g_num_conntol)"
	else # no connection fail or data too old (out of tolerance)
		if [ 0 -eq $g_defs_sent ] ; then
			g_defs_sending=1
			get_defaults
		fi
		if [ "$g_cycle" -gt "$g_num_smallcycles" ] ; then
			g_cycle=0
			if [[ 0 -eq $g_connfail && -f $g_floc_cfgfile ]] ; then # previous data were sent to server, rotate conf.backup file
				mv $g_floc_cfgfile $g_floc_oldcfgfile
			fi # keep old (last synced one) otherwise
			g_fsuff_var="$g_fsuff_big"
			g_output=$(run_big)
		else
			g_fsuff_var="$g_fsuff_small"
			g_output=$(run_small)
		fi
	fi
	m_pingfail=0
	m_connected=0
	while [ "$m_pingfail" -le "$g_num_pingtries" ] ; do
		if ping -c 1 "$g_serverurl" >$g_fn_devnull 2>&1 ; then
			m_connected=1
			break
		else # comm with server
			m_pingfail=$((m_pingfail+1))
			handle_me $g_loglvl_dbg "Failed to ping server $g_serverurl."
			isleep "$g_time_ping"
			continue
		fi
	done
	if [ 0 -eq "$m_connected" ] ; then # no connection after all attempts
		g_time_nextcyc=$g_time_fall
		g_connfail=$((g_connfail+1))
		dump_counters
		continue
	fi
	g_time_cycoverride=""
	# send and receive data now
	if data_exchange ; then
		# if defaults were sent, update g_defs_sent
		[ "$g_defs_sending" -eq 1 ] && {
			g_defs_sent=1
			g_defs_sending=0
			save_settings "file"
		}
		[ -f "$g_floc_cfgarch" ] && rm -f "$g_floc_cfgarch" 2>$g_fn_devnull
		[ 0 -lt $g_connfail ] && g_connfail=0 && dump_counters # connfail reset -> note to file
		g_connfail=0 # data exchanged anyway, set connfails to 0 anyway
		if [ -z "$g_time_cycoverride" ] ; then
			g_time_nextcyc=$g_time_reg
		else
			g_time_nextcyc=$g_time_cycoverride
		fi
	else
		g_connfail=$((g_connfail+1))
		load_settings
	fi
	if [[ "$g_tunstat" != "$g_tuns_none" && "$g_time_nextcyc" != "$g_ctime_atonce" ]]; then
		g_time_nextcyc=$g_time_tunnel
	fi
	[[ 1 -eq "$m_tunstat_cleanup" && "$g_tuns_stopped" = "$g_tunstat" ]] && {
		g_tunstat=$g_tuns_none
		m_tunstat_cleanup=0
		save_settings "file"
	}
	if [ -n "$g_todo_restart" ] ; then
		if [ "$g_tuns_none" -ne "$g_tunstat" ] ; then # stop tunnel
			read -r m_tpid <$g_floc_tunnpidfile
			if [ -d "/proc/$m_tpid" ] ; then
				kill "$m_tpid"
			fi
			g_tunstat=$g_tuns_stopped
			g_todo_tunnel="sure" # restart tunnel after reboot
		fi
		save_settings "file"
		handle_me $g_loglvl_warn "Either server or one of the performed operation requested reboot, rebooting..." $g_floc_glog
		sleep 1
		g_todo_restart=""
		reboot
	fi
	if [ -n "$g_todo_tunnel" ] ; then
		start_tunnel || handle_me $g_loglvl_err "Tunnel startup failed." "GLOG"
		g_todo_tunnel=""
		save_settings "file"
	fi
	[ 1 -eq "$g_num_loopend" ] && break
	[ 0 -eq "$g_num_loopend" ] || g_num_loopend=$((g_num_loopend-1))
	isleep "$g_time_nextcyc" "main"
done
process_cleanup "cycle_ends"
