<?php

declare(strict_types=1);

namespace App\Controller\Admin\Ajax;

use App\Entity\Device\Device;
use App\Entity\Tool\SimCard;
use App\Form\SimCardType;
use Doctrine\DBAL\Exception;
use Doctrine\DBAL\Exception\UniqueConstraintViolationException;
use Doctrine\Persistence\ManagerRegistry;
use Symfony\Bundle\FrameworkBundle\Controller\AbstractController as Controller;
use Symfony\Component\HttpFoundation\JsonResponse;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\HttpFoundation\Response;
use Symfony\Component\Routing\Annotation\Route;
use Symfony\Component\Validator\Validator\ValidatorInterface;

#[Route(path: '/admin/ajax/sim-card', name: 'ajax_sim_card_')]
class SimCardController extends Controller
{

    /**
     * Add new Sim card
     * @param Request $request
     * @param ManagerRegistry $managerRegistry
     * @param ValidatorInterface $validator
     * @return Response
     * @throws Exception
     */
    #[Route(path: '/add', name: 'add', methods: ['POST'])]
    public function addSimCardAction(Request $request, ManagerRegistry $managerRegistry, ValidatorInterface $validator): Response {

        $simCard = new SimCard();
        $devices = $managerRegistry->getRepository(Device::class)->getDevices();

        $form = $this->createForm(SimCardType::class, ['devices' => $devices]);
        $form->handleRequest($request);
        $errors = [];

        if(!$form->isValid()){

            $formErrors = $form->getErrors(true,true);

            foreach ($formErrors as $error){

                $errors[] = $error->getMessage();
            }
        }

        if ($form->isSubmitted()) {

            $errorsValidator = [];
            foreach ($errorsValidator as $error) {
                array_push($errors, $error->getMessage());
            }

            $requestData = $request->request->all()['sim_card'];
            $deviceId = (array_key_exists('device', $requestData)) ? $requestData['device'] : null;

            //Check Device if filled validation
            //$simCardFormData = $request->get('sim_card'); //does not work for device selected value
            /*if(!key_exists('device', $simCardFormData) || $simCardFormData['device'] === ''){
                array_push($errors, 'No device selected.');
            }*/

            if (count($errors) == 0) {

                $formData = $form->getData();
                $em = $managerRegistry->getManager();
                try {

                    $device = $managerRegistry->getRepository(Device::class)->findOneBy(['id' => $deviceId]);

                    $simCard->setIccid($formData['iccid']);
                    $simCard->setMsisdn($formData['msisdn']);
                    $simCard->setLabel($formData['label']);
                    $simCard->setDeviceUser($device);
                    $em->persist($simCard);
                    $em->flush();

                }catch (UniqueConstraintViolationException $e){

                    //dump($e->getMessage());

                    return new JsonResponse(['code' => 400, 'message' => 'error', 'errors' => ['error' => "Sim Card already exists."]],
                        400);

                }catch (\Exception $e){

                    return new JsonResponse(['code' => 400, 'message' => 'error', 'errors' => ['error' => "Unable to create the Sim Card."]],
                        400);

                }

                return new JsonResponse(['code' => 200, 'message' => 'success'],200);
            }

        }

        return new JsonResponse(['code' => 400, 'message' => 'error', 'errors' => $errors],
            400);
    }

    /**
     * Delete Sim Card
     * @param Request $request
     * @param int $simCardId
     * @param ManagerRegistry $managerRegistry
     * @return Response
     */
    #[Route(path: '/delete/{simCardId}', name: 'delete', methods: ['GET'])]
    public function labelDeleteAction(Request $request, int $simCardId, ManagerRegistry $managerRegistry): Response {

        if ($request->isXmlHttpRequest()) {

            $simCard = $managerRegistry->getRepository(SimCard::class)->findOneBy(['id' => $simCardId]);

            if($simCard){

                $em = $managerRegistry->getManager();
                $em->remove($simCard);
                $em->flush();

            }

            return new JsonResponse(['code' => 200, 'message' => 'Sim Card was deleted', 'errors' => ['errors' => ['']]],
                200);

        }

        return new Response('Bad request.', 400);

    }

}